// n1 Browser Agent - Side Panel Chat UI

const DEBUG_LOGGING = false;
function debugLog(...args) {
  if (DEBUG_LOGGING) console.log(...args);
}

const $ = (id) => document.getElementById(id);

// DOM Elements
const els = {
  settingsBtn: $('settings-btn'),
  settingsPanel: $('settings-panel'),
  mainPanel: $('main-panel'),
  apiKeyInput: $('api-key'),
  apiEndpointInput: $('api-endpoint'),
  saveSettingsBtn: $('save-settings'),
  cancelSettingsBtn: $('cancel-settings'),
  tasksDropdown: $('tasks-dropdown'),
  tasksDropdownBtn: $('tasks-dropdown-btn'),
  tasksDropdownLabel: $('tasks-dropdown-label'),
  tasksDropdownMenu: $('tasks-dropdown-menu'),
  chatContainer: $('chat-container'),
  welcomeMessage: $('welcome-message'),
  runningBar: $('running-bar'),
  runningStatus: $('running-status'),
  stepCount: $('step-count'),
  stopBtn: $('stop-btn'),
  taskInput: $('task-input'),
  sendBtn: $('send-btn')
};

// State
let currentTabId = null;
let lastRenderedState = null;
let dropdownOpen = false;
let renderedMessageIds = new Set(); // Track which messages are already in DOM

// Reusable SVG icons
const ICONS = {
  chevronDown: '<svg class="toggle-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="6 9 12 15 18 9"></polyline></svg>',
  thinking: '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><path d="M12 16v-4"/><path d="M12 8h.01"/></svg>'
};

// Completion status mappings
const COMPLETION_STATUS = {
  completed: { class: 'completed', icon: '✓', text: 'Completed' },
  stopped: { class: 'stopped', icon: '■', text: 'Stopped' },
  error: { class: 'error', icon: '✕', text: 'Error' },
  max_steps: { class: 'stopped', icon: '⏱', text: 'Max steps reached' }
};

// Get tabId from URL query parameter (set by background.js when opening)
function getTabIdFromUrl() {
  const params = new URLSearchParams(window.location.search);
  const tabId = params.get('tabId');
  return tabId ? parseInt(tabId, 10) : null;
}

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
  await loadSettings();

  // Get tabId from URL (more reliable than querying active tab)
  currentTabId = getTabIdFromUrl();

  // Fallback to querying active tab if not in URL
  if (!currentTabId) {
    await getCurrentTab();
  }

  // Establish port connection to background - used to detect panel close
  if (currentTabId) {
    const port = chrome.runtime.connect({ name: `sidepanel-${currentTabId}` });
    port.onDisconnect.addListener(() => {
      // Port disconnected - panel is closing
    });
  }

  // Clear any stale rendered state
  renderedMessageIds.clear();
  lastRenderedState = null;

  await refreshState();
  checkApiKey();
  setupEventListeners();
  setInterval(refreshState, 500);
});

function setupEventListeners() {
  // Settings
  els.settingsBtn.addEventListener('click', showSettings);
  els.saveSettingsBtn.addEventListener('click', saveSettings);
  els.cancelSettingsBtn.addEventListener('click', hideSettings);

  // Task input
  els.sendBtn.addEventListener('click', startTask);
  els.taskInput.addEventListener('keydown', (e) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      startTask();
    }
  });

  // Auto-resize textarea
  els.taskInput.addEventListener('input', () => {
    els.taskInput.style.height = 'auto';
    els.taskInput.style.height = Math.min(els.taskInput.scrollHeight, 120) + 'px';
  });

  // Stop button
  els.stopBtn.addEventListener('click', () => stopTaskByTabId(currentTabId));

  // Tasks dropdown
  els.tasksDropdownBtn.addEventListener('click', toggleDropdown);
  document.addEventListener('click', (e) => {
    if (!els.tasksDropdown.contains(e.target)) {
      closeDropdown();
    }
  });

  // Example buttons
  document.querySelectorAll('.example-btn').forEach(btn => {
    btn.addEventListener('click', () => {
      els.taskInput.value = btn.dataset.example;
      els.taskInput.focus();
      els.taskInput.style.height = 'auto';
      els.taskInput.style.height = Math.min(els.taskInput.scrollHeight, 120) + 'px';
    });
  });

  // Event delegation for dynamically created elements (like action toggles)
  els.chatContainer.addEventListener('click', (e) => {
    const toggleBtn = e.target.closest('[data-action="toggle-actions"]');
    if (toggleBtn) {
      const collapsedContainer = toggleBtn.closest('.actions-collapsed');
      if (collapsedContainer) {
        collapsedContainer.classList.toggle('expanded');
      }
    }
  });
}

async function getCurrentTab() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab) {
      currentTabId = tab.id;
    }
  } catch (e) {}
}

async function refreshState() {
  try {
    const state = await chrome.runtime.sendMessage({ type: 'GET_STATE' });
    updateUI(state);
  } catch (e) {
    console.error('[N1 UI] refreshState error:', e);
  }
}

function updateUI(state) {
  if (!state?.tasks) return;

  const { tasks, completedTasks = [] } = state;
  const runningTasks = tasks.filter(t => t.isRunning);
  let currentTabTask = tasks.find(t => t.tabId === currentTabId);

  // Check for a task in 'wrapping_up' or recently 'stopped' state (which is in completedTasks, not tasks)
  // This happens during the brief period when the task is stopped but summary is being fetched,
  // or just after the summary is fetched and we need to display it
  if (!currentTabTask) {
    // First look for wrapping_up, then for the most recent stopped task
    const wrappingUpTask = completedTasks.find(t => t.tabId === currentTabId && t.completionStatus === 'wrapping_up');
    if (wrappingUpTask) {
      currentTabTask = wrappingUpTask;
    } else {
      // Find the most recently completed task for this tab (to ensure transition completes properly)
      const recentStoppedTask = completedTasks
        .filter(t => t.tabId === currentTabId && t.completedAt)
        .sort((a, b) => (b.completedAt || 0) - (a.completedAt || 0))[0];
      if (recentStoppedTask) {
        currentTabTask = recentStoppedTask;
      }
    }
  }

  // Debug logging for stop flow (only on state changes to reduce spam)
  const tabCompleted = completedTasks.filter(t => t.tabId === currentTabId);
  if (tabCompleted.length > 0 || currentTabTask) {
    debugLog('[N1 UI] updateUI - currentTabTask:', currentTabTask?.completionStatus, 'running:', currentTabTask?.isRunning);
    debugLog('[N1 UI] updateUI - completedTasks for tab:', tabCompleted.length, tabCompleted.map(t => ({ status: t.completionStatus, answer: t.finalAnswer?.substring(0, 50) })));
  }

  // Update tasks dropdown
  updateTasksDropdown(runningTasks);

  // Update running bar - treat 'wrapping_up' as running (waiting for summary)
  const isWrappingUp = currentTabTask?.completionStatus === 'wrapping_up';
  const isEffectivelyRunning = currentTabTask?.isRunning || isWrappingUp;
  if (isEffectivelyRunning) {
    toggleClass(els.runningBar, 'hidden', false);
    els.stepCount.textContent = `Step ${currentTabTask.stepCount}`;
    if (isWrappingUp) {
      els.runningStatus.textContent = 'Getting summary...';
      // Disable stop button during wrapping up (already stopping)
      els.stopBtn.disabled = true;
      els.stopBtn.textContent = 'Stopping...';
    } else {
      els.runningStatus.textContent = currentTabTask.stepCount === 0 ? 'Starting...' : 'Running';
      els.stopBtn.disabled = false;
      els.stopBtn.textContent = 'Stop';
    }
  } else {
    toggleClass(els.runningBar, 'hidden', true);
    // Reset stop button state when not running
    els.stopBtn.disabled = false;
    els.stopBtn.textContent = 'Stop';
  }

  // Disable task input while a task is running
  if (isEffectivelyRunning) {
    els.taskInput.disabled = true;
    els.taskInput.placeholder = 'Task in progress...';
    els.sendBtn.disabled = true;
  } else {
    els.taskInput.disabled = false;
    els.taskInput.placeholder = 'What would you like me to do?';
    els.sendBtn.disabled = false;
  }

  // Build chat history for current tab
  const chatHistory = buildChatHistory(currentTabTask, completedTasks);

  // Use incremental rendering instead of full re-render
  renderChatIncremental(chatHistory, currentTabTask);
}

function buildChatHistory(currentTask, completedTasks) {
  const history = [];

  // Get the current task's startedAt to exclude it from completed tasks (avoid duplicates)
  const currentTaskStartedAt = currentTask?.startedAt;

  // Add completed tasks for this tab (in chronological order)
  // Exclude the current task if it appears in both lists
  const tabCompletedTasks = completedTasks
    .filter(t => t.tabId === currentTabId && t.startedAt !== currentTaskStartedAt)
    .sort((a, b) => (a.completedAt || 0) - (b.completedAt || 0));

  if (tabCompletedTasks.length > 0) {
    debugLog('[N1 UI] buildChatHistory - tabCompletedTasks:', tabCompletedTasks.map(t => ({
      startedAt: t.startedAt,
      status: t.completionStatus,
      answer: t.finalAnswer?.substring(0, 30)
    })));
  }

  for (const task of tabCompletedTasks) {
    history.push(...buildTaskMessages(task, false));
  }

  // Add current task (whether running or just completed)
  if (currentTask) {
    history.push(...buildTaskMessages(currentTask, currentTask.isRunning));
  }

  return history;
}

function buildTaskMessages(task, isRunning) {
  const messages = [];
  const taskId = task.startedAt || Date.now();

  // Treat 'wrapping_up' status as still running - we're waiting for the final summary
  const isEffectivelyRunning = isRunning || task.completionStatus === 'wrapping_up';

  // User message (the task)
  if (task.currentTask) {
    messages.push({
      id: `user-${taskId}`,
      type: 'user',
      content: task.currentTask
    });
  }

  // Assistant response from action log
  if (task.actionLog?.length > 0) {
    const assistantMsg = buildAssistantMessage(task.actionLog, isEffectivelyRunning, task.completionStatus, task.finalAnswer);
    if (assistantMsg) {
      assistantMsg.id = isEffectivelyRunning ? `running-${taskId}` : `assistant-${taskId}`;
      messages.push(assistantMsg);
    }
  } else if (isEffectivelyRunning) {
    // Show thinking indicator when no actions yet
    messages.push({
      id: `running-${taskId}`,
      type: 'assistant',
      isThinking: true,
      thoughts: [],
      actions: [],
      answer: null,
      status: null,
      isRunning: true
    });
  }

  return messages;
}

function buildAssistantMessage(actionLog, isRunning, completionStatus, finalAnswer) {
  const thoughts = [];
  const actions = [];
  let answer = null;
  let error = null;
  let status = null;
  let lastErrorIndex = -1;
  let lastSuccessIndex = -1;

  for (let i = 0; i < actionLog.length; i++) {
    const entry = actionLog[i];
    switch (entry.type) {
      case 'thoughts':
        thoughts.push(entry.content);
        lastSuccessIndex = i; // Thoughts indicate successful API response
        break;
      case 'action':
        actions.push({
          type: entry.content,
          details: entry.details
        });
        lastSuccessIndex = i; // Actions indicate successful execution
        break;
      case 'complete':
        answer = entry.content;
        lastSuccessIndex = i;
        break;
      case 'error':
        error = entry.content;
        lastErrorIndex = i;
        break;
      case 'status':
        status = entry.content;
        break;
    }
  }

  // Only show error if it's the most recent significant event
  // (i.e., no successful thoughts/actions/completion after the error)
  if (lastSuccessIndex > lastErrorIndex) {
    error = null; // Clear error since model has continued successfully
  }

  // Use final answer if available (from completed task)
  if (finalAnswer) {
    answer = finalAnswer;
  }

  return {
    type: 'assistant',
    thoughts,
    actions,
    answer,
    error,
    status,
    isRunning,
    completionStatus
  };
}

function renderChatIncremental(history, currentTask) {
  // Show/hide welcome message
  toggleClass(els.welcomeMessage, 'hidden', history.length > 0);

  let shouldScroll = false;

  // FIRST: Handle task completion transition (running -> completed)
  // This must happen before processing history to avoid duplicates
  // Don't transition while in 'wrapping_up' state - wait for the final summary
  const runningEl = els.chatContainer.querySelector('[data-msg-id^="running-"]');
  const isStillWrappingUp = currentTask?.completionStatus === 'wrapping_up';
  if (runningEl && (!currentTask || (!currentTask.isRunning && !isStillWrappingUp))) {
    const oldId = runningEl.getAttribute('data-msg-id');
    const taskId = oldId.replace('running-', '');
    const newId = `assistant-${taskId}`;
    debugLog('[N1 UI] Transitioning running to completed, oldId:', oldId, 'newId:', newId);
    runningEl.setAttribute('data-msg-id', newId);
    renderedMessageIds.delete(oldId);
    renderedMessageIds.add(newId);

    // Find and apply the completed message state
    let completedMsg = history.find(m => m.id === newId);
    debugLog('[N1 UI] Found completedMsg:', !!completedMsg, completedMsg?.completionStatus, completedMsg?.answer?.substring(0, 50));

    // Fallback: if not in history, try to build from currentTask directly
    if (!completedMsg && currentTask && !currentTask.isRunning) {
      debugLog('[N1 UI] Building completedMsg from currentTask fallback');
      const taskMessages = buildTaskMessages(currentTask, false);
      completedMsg = taskMessages.find(m => m.type === 'assistant');
    }

    if (completedMsg) {
      updateRunningMessageInPlace(runningEl, completedMsg, true);
    } else {
      debugLog('[N1 UI] WARNING: No completed message found! History ids:', history.map(m => m.id));
    }
  }

  for (const msg of history) {
    const msgId = msg.id;
    if (!msgId) continue;

    const existingEl = els.chatContainer.querySelector(`[data-msg-id="${msgId}"]`);

    if (msg.isRunning) {
      if (existingEl) {
        // Update running message incrementally (preserving UI state like expanded)
        shouldScroll = updateRunningMessageInPlace(existingEl, msg) || shouldScroll;
      } else {
        // Create new running message
        const el = createMessageElement(msg);
        el.setAttribute('data-msg-id', msgId);
        els.chatContainer.appendChild(el);
        renderedMessageIds.add(msgId);
        shouldScroll = true;
      }
    } else if (!renderedMessageIds.has(msgId)) {
      // Completed message not yet rendered - append it
      // But check if there's a running version we should convert instead
      const taskId = msgId.replace('assistant-', '');
      const existingRunning = els.chatContainer.querySelector(`[data-msg-id="running-${taskId}"]`);
      if (existingRunning) {
        // Convert running to completed instead of creating new
        existingRunning.setAttribute('data-msg-id', msgId);
        renderedMessageIds.delete(`running-${taskId}`);
        renderedMessageIds.add(msgId);
        updateRunningMessageInPlace(existingRunning, msg, true);
        shouldScroll = true;
        continue;
      }

      const el = createMessageElement(msg);
      el.setAttribute('data-msg-id', msgId);

      // Insert before any running message
      const runningMsgEl = els.chatContainer.querySelector('[data-msg-id^="running-"]');
      if (runningMsgEl) {
        els.chatContainer.insertBefore(el, runningMsgEl);
      } else {
        els.chatContainer.appendChild(el);
      }
      renderedMessageIds.add(msgId);
      shouldScroll = true;
    } else if (existingEl && msg.answer) {
      // Fix race condition: Update completed messages that were rendered without a summary
      // This can happen if the message was rendered before finalAnswer was set
      const hasSummary = existingEl.querySelector('.summary-section');
      if (!hasSummary) {
        debugLog('[N1 UI] Updating completed message with missing summary, answer:', msg.answer?.substring(0, 50));
        updateRunningMessageInPlace(existingEl, msg, true);
        shouldScroll = true;
      }
    }
    // Note: Completed messages with summaries are NOT updated to preserve UI state.
  }

  // Scroll to bottom only when new content added
  if (shouldScroll) {
    els.chatContainer.scrollTop = els.chatContainer.scrollHeight;
  }
}

// Update a running message in place without replacing the entire element
function updateRunningMessageInPlace(el, msg, isCompletion = false) {
  let changed = false;
  let bubble = el.querySelector('.message-bubble');
  const isCompleted = msg.completionStatus && !msg.isRunning;

  // For completed tasks with an answer, rebuild with summary-focused view
  if (isCompletion && isCompleted && msg.answer) {
    // Replace entire bubble with completed view
    el.innerHTML = createAssistantBubble(msg);
    return true;
  }

  // Create bubble if it doesn't exist
  if (!bubble) {
    bubble = document.createElement('div');
    bubble.className = 'message-bubble';
    el.appendChild(bubble);
  }

  // Update thinking section
  changed = updateThinkingSection(bubble, msg) || changed;

  // Update actions section (append new actions, preserve expanded state)
  changed = updateActionsSection(bubble, msg) || changed;

  // Update status indicator
  changed = updateStatusIndicator(bubble, msg) || changed;

  // Update error section
  changed = updateErrorSection(bubble, msg) || changed;

  // Update answer section
  changed = updateAnswerSection(bubble, msg) || changed;

  // Update completion badge (only on completion)
  if (isCompletion || msg.completionStatus) {
    changed = updateCompletionBadge(bubble, msg) || changed;
  }

  return changed;
}

function updateThinkingSection(bubble, msg) {
  let changed = false;
  const isCompleted = msg.completionStatus && !msg.isRunning;

  // For completed tasks with answers, hide thinking section entirely
  if (isCompleted && msg.answer) {
    const thinkingSection = bubble.querySelector('.thinking-section');
    const analyzingIndicator = bubble.querySelector('.analyzing-indicator');
    if (thinkingSection) { thinkingSection.remove(); changed = true; }
    if (analyzingIndicator) { analyzingIndicator.remove(); changed = true; }
    return changed;
  }

  // Handle "Analyzing" indicator (shown when running with no thoughts/actions)
  let analyzingIndicator = bubble.querySelector('.analyzing-indicator');
  const shouldShowAnalyzing = msg.isRunning && !msg.thoughts?.length && !msg.actions?.length;

  if (shouldShowAnalyzing && !analyzingIndicator) {
    analyzingIndicator = document.createElement('div');
    analyzingIndicator.className = 'assistant-section analyzing-indicator';
    analyzingIndicator.innerHTML = `
      <div class="status-indicator">
        <div class="mini-pulse"></div>
        <span>Analyzing the page...</span>
      </div>
    `;
    bubble.insertBefore(analyzingIndicator, bubble.firstChild);
    changed = true;
  } else if (!shouldShowAnalyzing && analyzingIndicator) {
    analyzingIndicator.remove();
    changed = true;
  }

  // Handle thinking section
  let section = bubble.querySelector('.thinking-section');
  const shouldShow = msg.thoughts?.length > 0;
  const latestThought = msg.thoughts?.[msg.thoughts.length - 1];

  if (shouldShow) {
    if (!section) {
      section = document.createElement('div');
      section.className = 'assistant-section thinking-section';
      section.innerHTML = `
        <div class="thinking-header">
          ${ICONS.thinking}
          Thinking
        </div>
        <div class="thinking-content"></div>
      `;
      // Insert after analyzing indicator if present, otherwise at start
      const insertAfter = bubble.querySelector('.analyzing-indicator');
      if (insertAfter) {
        insertAfter.after(section);
      } else {
        bubble.insertBefore(section, bubble.firstChild);
      }
      changed = true;
    }
    const content = section.querySelector('.thinking-content');
    if (content && content.textContent !== latestThought) {
      content.textContent = latestThought;
      changed = true;
    }
  }

  return changed;
}

function updateActionsSection(bubble, msg) {
  let section = bubble.querySelector('.actions-section');
  const actions = msg.actions || [];

  if (actions.length === 0) {
    if (section) section.remove();
    return false;
  }

  if (!section) {
    section = document.createElement('div');
    section.className = 'assistant-section actions-section';
    // Insert after thinking section or at the start
    const thinkingSection = bubble.querySelector('.thinking-section');
    const analyzingIndicator = bubble.querySelector('.analyzing-indicator');
    const insertAfter = thinkingSection || analyzingIndicator;
    if (insertAfter) {
      insertAfter.after(section);
    } else {
      bubble.insertBefore(section, bubble.firstChild);
    }
  }

  // Get current rendered action count
  const renderedActions = section.querySelectorAll('.action-item');
  const renderedCount = renderedActions.length;

  if (actions.length > renderedCount) {
    // New actions to add
    const collapsedContainer = section.querySelector('.actions-collapsed');
    const isExpanded = collapsedContainer?.classList.contains('expanded');

    // Rebuild the section structure but preserve expanded state
    section.innerHTML = '';

    if (actions.length > 1) {
      const hiddenCount = actions.length - 1;
      const collapsed = document.createElement('div');
      collapsed.className = 'actions-collapsed' + (isExpanded ? ' expanded' : '');
      collapsed.innerHTML = `
        <button class="actions-toggle-btn" data-action="toggle-actions">
          ${ICONS.chevronDown}
          <span class="toggle-text">${hiddenCount} previous action${hiddenCount > 1 ? 's' : ''}</span>
        </button>
        <div class="actions-hidden"></div>
      `;
      const hiddenContainer = collapsed.querySelector('.actions-hidden');
      for (let i = 0; i < hiddenCount; i++) {
        hiddenContainer.insertAdjacentHTML('beforeend', createActionItem(actions[i]));
      }
      section.appendChild(collapsed);
    }

    // Add the latest action
    section.insertAdjacentHTML('beforeend', createActionItem(actions[actions.length - 1]));
    return true;
  }

  return false;
}

function updateStatusIndicator(bubble, msg) {
  let indicator = bubble.querySelector('.working-indicator');
  const isWrappingUp = msg.completionStatus === 'wrapping_up';
  const shouldShow = msg.isRunning && (msg.thoughts?.length > 0 || msg.actions?.length > 0 || isWrappingUp);
  const statusText = isWrappingUp ? 'Getting summary...' : 'Working...';

  if (shouldShow) {
    if (!indicator) {
      indicator = document.createElement('div');
      indicator.className = 'assistant-section working-indicator';
      indicator.innerHTML = `
        <div class="status-indicator">
          <div class="mini-pulse"></div>
          <span>${statusText}</span>
        </div>
      `;
      // Insert before error/answer sections
      const errorSection = bubble.querySelector('.error-section');
      const answerSection = bubble.querySelector('.answer-section');
      const insertBefore = errorSection || answerSection;
      if (insertBefore) {
        bubble.insertBefore(indicator, insertBefore);
      } else {
        bubble.appendChild(indicator);
      }
      return true;
    } else {
      // Update text if it changed (e.g., from "Working..." to "Getting summary...")
      const span = indicator.querySelector('span');
      if (span && span.textContent !== statusText) {
        span.textContent = statusText;
        return true;
      }
    }
  } else if (indicator) {
    indicator.remove();
    return true;
  }
  return false;
}

function updateErrorSection(bubble, msg) {
  let section = bubble.querySelector('.error-section');

  if (msg.error) {
    if (!section) {
      section = document.createElement('div');
      section.className = 'assistant-section error-section';
      bubble.appendChild(section);
    }
    if (section.textContent !== msg.error) {
      section.textContent = msg.error;
      return true;
    }
  } else if (section) {
    section.remove();
    return true;
  }
  return false;
}

function updateAnswerSection(bubble, msg) {
  let section = bubble.querySelector('.answer-section');
  const isCompleted = msg.completionStatus && !msg.isRunning;

  // Skip for completed tasks - they use summary-section instead
  if (isCompleted) {
    if (section) section.remove();
    return false;
  }

  if (msg.answer) {
    if (!section) {
      section = document.createElement('div');
      section.className = 'assistant-section answer-section';
      section.innerHTML = '<div class="answer-content"></div>';
      bubble.appendChild(section);
    }
    const content = section.querySelector('.answer-content');
    const renderedAnswer = renderMarkdown(msg.answer);
    if (content && content.innerHTML !== renderedAnswer) {
      content.innerHTML = renderedAnswer;
      return true;
    }
  } else if (section) {
    section.remove();
    return true;
  }
  return false;
}

function updateCompletionBadge(bubble, msg) {
  let badgeContainer = bubble.querySelector('.completion-badge-container');

  if (msg.completionStatus && !msg.isRunning) {
    const statusInfo = COMPLETION_STATUS[msg.completionStatus] || { class: 'stopped', icon: '•', text: msg.completionStatus };

    if (!badgeContainer) {
      badgeContainer = document.createElement('div');
      badgeContainer.className = 'assistant-section completion-badge-container';
      badgeContainer.style.padding = '8px 14px';
      bubble.appendChild(badgeContainer);
    }

    const newBadgeHtml = `<span class="completion-badge ${statusInfo.class}">${statusInfo.icon} ${statusInfo.text}</span>`;
    if (badgeContainer.innerHTML !== newBadgeHtml) {
      badgeContainer.innerHTML = newBadgeHtml;
      return true;
    }
  } else if (badgeContainer) {
    badgeContainer.remove();
    return true;
  }
  return false;
}

function createMessageElement(msg) {
  const el = document.createElement('div');
  el.className = `chat-message ${msg.type}`;

  if (msg.type === 'user') {
    el.innerHTML = `<div class="message-content">${escapeHtml(msg.content)}</div>`;
  } else if (msg.type === 'assistant') {
    el.innerHTML = createAssistantBubble(msg);
  } else if (msg.type === 'system') {
    el.innerHTML = `<div class="message-content">${escapeHtml(msg.content)}</div>`;
  }

  return el;
}

// Simple markdown renderer for summaries
function renderMarkdown(text) {
  if (!text) return '';

  let html = escapeHtml(text);

  // Convert headers (must be at start of line)
  html = html.replace(/^### (.+)$/gm, '<h4>$1</h4>');
  html = html.replace(/^## (.+)$/gm, '<h3>$1</h3>');
  html = html.replace(/^# (.+)$/gm, '<h2>$1</h2>');

  // Convert bold **text** and __text__
  html = html.replace(/\*\*([^*]+)\*\*/g, '<strong>$1</strong>');
  html = html.replace(/__([^_]+)__/g, '<strong>$1</strong>');

  // Convert italic *text* and _text_ (but not inside words)
  html = html.replace(/(?<![*\w])\*([^*]+)\*(?![*\w])/g, '<em>$1</em>');
  html = html.replace(/(?<![_\w])_([^_]+)_(?![_\w])/g, '<em>$1</em>');

  // Convert inline code `code`
  html = html.replace(/`([^`]+)`/g, '<code>$1</code>');

  // Convert unordered lists (- item or * item)
  html = html.replace(/^[-*] (.+)$/gm, '<li>$1</li>');

  // Convert numbered lists (1. item)
  html = html.replace(/^\d+\. (.+)$/gm, '<li>$1</li>');

  // Wrap consecutive <li> in <ul>
  html = html.replace(/(<li>[\s\S]*?<\/li>)(?=\s*(?:<li>|$))/g, (match, content, offset, string) => {
    const before = string.substring(0, offset);
    const isFirstInList = !before.endsWith('</li>\n') && !before.endsWith('</li>');
    return isFirstInList ? '<ul>' + content : content;
  });
  html = html.replace(/(<\/li>)(?!\s*<li>)/g, '$1</ul>');

  // Convert line breaks to <br> (but not after block elements)
  html = html.replace(/\n(?!<\/?(?:h[1-6]|ul|li|p))/g, '<br>');

  // Clean up multiple <br>
  html = html.replace(/(<br>){3,}/g, '<br><br>');

  return html;
}

function createAssistantBubble(msg) {
  const isCompleted = msg.completionStatus && !msg.isRunning;
  let html = '<div class="message-bubble">';

  // For completed tasks with an answer, show a cleaner summary-focused view
  if (isCompleted && msg.answer) {
    // Show answer/summary as the main content with markdown
    html += `
      <div class="assistant-section summary-section">
        <div class="summary-content">${renderMarkdown(msg.answer)}</div>
      </div>
    `;

    // Collapsible actions history (all collapsed by default)
    if (msg.actions?.length > 0) {
      html += `
        <div class="assistant-section actions-section actions-summary">
          <div class="actions-collapsed">
            <button class="actions-toggle-btn" data-action="toggle-actions">
              ${ICONS.chevronDown}
              <span class="toggle-text">${msg.actions.length} action${msg.actions.length > 1 ? 's' : ''} taken</span>
            </button>
            <div class="actions-hidden">
      `;
      for (const action of msg.actions) {
        html += createActionItem(action);
      }
      html += `
            </div>
          </div>
        </div>
      `;
    }
  } else {
    // Running task view - show thinking and actions

    // Thinking section (only for running tasks)
    if (msg.thoughts?.length > 0) {
      const latestThought = msg.thoughts[msg.thoughts.length - 1];
      html += `
        <div class="assistant-section thinking-section">
          <div class="thinking-header">
            ${ICONS.thinking}
            Thinking
          </div>
          <div class="thinking-content">${escapeHtml(latestThought)}</div>
        </div>
      `;
    } else if (msg.isRunning && !msg.actions?.length) {
      // Show analyzing indicator when starting
      html += `
        <div class="assistant-section analyzing-indicator">
          <div class="status-indicator">
            <div class="mini-pulse"></div>
            <span>Analyzing the page...</span>
          </div>
        </div>
      `;
    }

    // Actions section (collapsible - show only last action by default)
    if (msg.actions?.length > 0) {
      const actionCount = msg.actions.length;
      const hiddenCount = actionCount - 1;

      html += '<div class="assistant-section actions-section">';

      if (hiddenCount > 0) {
        html += `
          <div class="actions-collapsed">
            <button class="actions-toggle-btn" data-action="toggle-actions">
              ${ICONS.chevronDown}
              <span class="toggle-text">${hiddenCount} previous action${hiddenCount > 1 ? 's' : ''}</span>
            </button>
            <div class="actions-hidden">
        `;
        for (let i = 0; i < hiddenCount; i++) {
          html += createActionItem(msg.actions[i]);
        }
        html += `
            </div>
          </div>
        `;
      }

      html += createActionItem(msg.actions[actionCount - 1]);
      html += '</div>';
    }

    // Running indicator
    if (msg.isRunning && (msg.thoughts?.length > 0 || msg.actions?.length > 0)) {
      html += `
        <div class="assistant-section working-indicator">
          <div class="status-indicator">
            <div class="mini-pulse"></div>
            <span>Working...</span>
          </div>
        </div>
      `;
    }

    // Error section
    if (msg.error) {
      html += `
        <div class="assistant-section error-section">
          ${escapeHtml(msg.error)}
        </div>
      `;
    }

    // Answer section (for running tasks that receive partial answers)
    if (msg.answer && !isCompleted) {
      html += `
        <div class="assistant-section answer-section">
          <div class="answer-content">${renderMarkdown(msg.answer)}</div>
        </div>
      `;
    }
  }

  // Completion badge (for completed tasks)
  if (msg.completionStatus && !msg.isRunning) {
    const badgeClass = {
      completed: 'completed',
      stopped: 'stopped',
      error: 'error',
      max_steps: 'stopped'
    }[msg.completionStatus] || 'stopped';

    const badgeIcon = {
      completed: '✓',
      stopped: '■',
      error: '✕',
      max_steps: '⏱'
    }[msg.completionStatus] || '•';

    const badgeText = {
      completed: 'Completed',
      stopped: 'Stopped',
      error: 'Error',
      max_steps: 'Max steps reached'
    }[msg.completionStatus] || msg.completionStatus;

    html += `
      <div class="assistant-section" style="padding: 8px 14px;">
        <span class="completion-badge ${badgeClass}">${badgeIcon} ${badgeText}</span>
      </div>
    `;
  }

  html += '</div>';
  return html;
}

function createActionItem(action) {
  const iconSvg = getActionIcon(action);
  const details = formatActionDetails(action.details);

  return `
    <div class="action-item">
      <div class="action-icon">${iconSvg}</div>
      <div class="action-text">
        <span class="action-type">${escapeHtml(action.type)}</span>
        ${details ? `<span class="action-details">${escapeHtml(details)}</span>` : ''}
      </div>
    </div>
  `;
}

function getActionIcon(action) {
  const type = (action?.type || '').toLowerCase();
  const details = action?.details || {};

  if (type.includes('click') || type.includes('tap')) {
    return '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M15 15l-2 5L9 9l11 4-5 2z"/></svg>';
  }
  if (type.includes('type') || type.includes('text') || type.includes('key')) {
    return '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="2" y="4" width="20" height="16" rx="2"/><path d="M6 8h.01M10 8h.01M14 8h.01M18 8h.01M8 12h.01M12 12h.01M16 12h.01M7 16h10"/></svg>';
  }
  if (type.includes('scroll')) {
    const direction = (details.direction || 'down').toLowerCase();
    if (direction === 'up') {
      return '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 19V5M5 12l7-7 7 7"/></svg>';
    } else if (direction === 'left') {
      return '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M19 12H5M12 5l-7 7 7 7"/></svg>';
    } else if (direction === 'right') {
      return '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M5 12h14M12 5l7 7-7 7"/></svg>';
    }
    // Default: down
    return '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 5v14M5 12l7 7 7-7"/></svg>';
  }
  if (type.includes('drag')) {
    return '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M5 9l7-7 7 7M5 15l7 7 7-7"/></svg>';
  }
  if (type.includes('goto') || type.includes('navigate') || type.includes('url') || type === 'back' || type === 'go_back') {
    return '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"/><polyline points="15 3 21 3 21 9"/><line x1="10" y1="14" x2="21" y2="3"/></svg>';
  }
  if (type.includes('wait')) {
    return '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><polyline points="12 6 12 12 16 14"/></svg>';
  }

  // Default action icon
  return '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="9 11 12 14 22 4"/><path d="M21 12v7a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11"/></svg>';
}

function formatActionDetails(details) {
  if (!details) return '';
  const parts = [];
  const actionType = details.action_type || details.type || details.name;

  // Support both API format (coordinates) and legacy format (center_coordinates)
  const coords = details.coordinates || details.center_coordinates;
  const startCoords = details.start_coordinates || details.startCoordinates;
  const endCoords = details.end_coordinates || details.endCoordinates || coords;

  if (actionType === 'drag' && startCoords) {
    if (endCoords) {
      parts.push(`from (${startCoords[0]}, ${startCoords[1]}) to (${endCoords[0]}, ${endCoords[1]})`);
    }
  } else if (coords) {
    parts.push(`at (${coords[0]}, ${coords[1]})`);
  }

  if (details.text) parts.push(`"${details.text}"`);
  if (details.direction) parts.push(details.direction);
  if (details.url) parts.push(details.url);
  // Support both keys array and key_comb string
  if (details.keys) parts.push(details.keys.join('+'));
  else if (details.key_comb) parts.push(details.key_comb);
  else if (details.key) parts.push(details.key);
  return parts.join(' ');
}

function updateTasksDropdown(runningTasks) {
  // Show dropdown only when there are running tasks on OTHER tabs
  const otherTabTasks = runningTasks.filter(t => t.tabId !== currentTabId);

  if (otherTabTasks.length === 0) {
    toggleClass(els.tasksDropdown, 'hidden', true);
    return;
  }

  toggleClass(els.tasksDropdown, 'hidden', false);
  els.tasksDropdownLabel.textContent = `${otherTabTasks.length} other ${otherTabTasks.length === 1 ? 'task' : 'tasks'}`;

  // Render dropdown items
  els.tasksDropdownMenu.innerHTML = otherTabTasks.map(task => {
    const title = task.tabTitle ? truncate(task.tabTitle, 25) : `Tab ${task.tabId}`;
    return `
      <div class="dropdown-item" data-tab-id="${task.tabId}">
        <div class="dropdown-item-info">
          <div class="dropdown-item-title">${escapeHtml(title)}</div>
          <div class="dropdown-item-meta">
            <span class="dropdown-item-step">Step ${task.stepCount}</span>
            <span>${escapeHtml(truncate(task.currentTask || '', 20))}</span>
          </div>
        </div>
        <button class="dropdown-item-stop" data-tab-id="${task.tabId}">Stop</button>
      </div>
    `;
  }).join('');

  // Attach event listeners
  els.tasksDropdownMenu.querySelectorAll('.dropdown-item').forEach(item => {
    item.addEventListener('click', async (e) => {
      if (!e.target.classList.contains('dropdown-item-stop')) {
        const tabId = parseInt(item.dataset.tabId);
        try {
          await chrome.tabs.update(tabId, { active: true });
          const tab = await chrome.tabs.get(tabId);
          if (tab.windowId) await chrome.windows.update(tab.windowId, { focused: true });
        } catch (e) {}
        closeDropdown();
      }
    });
  });

  els.tasksDropdownMenu.querySelectorAll('.dropdown-item-stop').forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      stopTaskByTabId(parseInt(btn.dataset.tabId));
    });
  });
}

function toggleDropdown() {
  dropdownOpen = !dropdownOpen;
  toggleClass(els.tasksDropdownMenu, 'hidden', !dropdownOpen);
  els.tasksDropdownBtn.classList.toggle('open', dropdownOpen);
}

function closeDropdown() {
  dropdownOpen = false;
  toggleClass(els.tasksDropdownMenu, 'hidden', true);
  els.tasksDropdownBtn.classList.remove('open');
}

function toggleClass(el, className, condition) {
  el.classList.toggle(className, condition);
}

function truncate(str, maxLen) {
  return str.length <= maxLen ? str : str.substring(0, maxLen - 3) + '...';
}

function escapeHtml(text) {
  if (!text) return '';
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

// Settings
async function loadSettings() {
  const { apiKey, apiEndpoint } = await chrome.storage.local.get(['apiKey', 'apiEndpoint']);
  if (apiKey) els.apiKeyInput.value = apiKey;
  if (apiEndpoint) els.apiEndpointInput.value = apiEndpoint;
}

async function saveSettings() {
  const apiKey = els.apiKeyInput.value.trim();
  const apiEndpoint = els.apiEndpointInput.value.trim();

  if (!apiKey) {
    alert('Please enter an API key');
    return;
  }

  await chrome.storage.local.set({ apiKey, apiEndpoint });
  hideSettings();
  checkApiKey();
}

function checkApiKey() {
  chrome.storage.local.get(['apiKey'], (result) => {
    if (!result.apiKey) showSettings();
  });
}

function showSettings() {
  toggleClass(els.settingsPanel, 'hidden', false);
}

function hideSettings() {
  toggleClass(els.settingsPanel, 'hidden', true);
}

// Task Management
async function startTask() {
  const task = els.taskInput.value.trim();
  if (!task) return;

  const { apiKey } = await chrome.storage.local.get(['apiKey']);
  if (!apiKey) {
    showSettings();
    return;
  }

  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab.url || tab.url.startsWith('chrome://') || tab.url.startsWith('chrome-extension://')) {
      alert('Cannot run on Chrome system pages.\n\nPlease navigate to any website first.');
      return;
    }

    // Clear input immediately for better UX
    els.taskInput.value = '';
    els.taskInput.style.height = 'auto';
    lastRenderedState = null; // Force re-render

    currentTabId = tab.id;
    const response = await chrome.runtime.sendMessage({ type: 'START_TASK', task, tabId: tab.id });
    if (response?.error) {
      alert('Error: ' + response.error);
    }
  } catch (error) {
    alert('Error: ' + error.message);
  }
}

async function stopTaskByTabId(tabId) {
  try {
    await chrome.runtime.sendMessage({ type: 'STOP_TASK', tabId });
  } catch (e) {}
}
