// n1 Browser Agent - Popup UI

const $ = (id) => document.getElementById(id);

// DOM Elements
const els = {
  settingsBtn: $('settings-btn'),
  settingsPanel: $('settings-panel'),
  mainPanel: $('main-panel'),
  apiKeyInput: $('api-key'),
  apiEndpointInput: $('api-endpoint'),
  saveSettingsBtn: $('save-settings'),
  cancelSettingsBtn: $('cancel-settings'),
  activeTasksSection: $('active-tasks-section'),
  activeTasksList: $('active-tasks-list'),
  taskCount: $('task-count'),
  currentTabSection: $('current-tab-section'),
  currentTabStatus: $('current-tab-status'),
  inputSection: $('input-section'),
  runningSection: $('running-section'),
  taskInput: $('task-input'),
  startBtn: $('start-btn'),
  stopBtn: $('stop-btn'),
  stepCount: $('step-count'),
  currentTask: $('current-task'),
  logContainer: $('log-container'),
  logTitle: $('log-title'),
  welcomeMessage: $('welcome-message'),
  clearBtn: $('clear-btn'),
  completedTasksSection: $('completed-tasks-section'),
  completedTasksList: $('completed-tasks-list'),
  clearCompletedBtn: $('clear-completed-btn')
};

// State
let currentTabId = null;
let lastLogLength = 0;
let lastRenderedTaskTabId = null;
let lastRenderedTaskStartedAt = null;
let selectedTaskTabId = null;
let expandedCompletedTasks = new Set();
let lastCompletedTasksHash = null;

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
  await loadSettings();
  await getCurrentTab();
  await refreshState();
  checkApiKey();
  setInterval(refreshState, 500);
});

async function getCurrentTab() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab) {
      currentTabId = tab.id;
      selectedTaskTabId = tab.id;
    }
  } catch (e) {}
}

async function refreshState() {
  try {
    const state = await chrome.runtime.sendMessage({ type: 'GET_STATE' });
    updateUI(state);
  } catch (e) {}
}

function updateUI(state) {
  if (!state?.tasks) return;

  const { tasks, completedTasks = [] } = state;
  const runningTasks = tasks.filter(t => t.isRunning);
  const currentTabTask = tasks.find(t => t.tabId === currentTabId);

  // Active tasks section
  toggleClass(els.activeTasksSection, 'hidden', runningTasks.length === 0);
  if (runningTasks.length > 0) {
    els.taskCount.textContent = runningTasks.length;
    renderActiveTasks(runningTasks);
  }

  // Completed tasks section
  toggleClass(els.completedTasksSection, 'hidden', completedTasks.length === 0);
  if (completedTasks.length > 0) {
    renderCompletedTasks(completedTasks);
  }

  // Current tab section
  if (currentTabTask?.isRunning) {
    toggleClass(els.inputSection, 'hidden', true);
    toggleClass(els.runningSection, 'hidden', false);
    els.currentTask.textContent = currentTabTask.currentTask || '';
    els.stepCount.textContent = `Step ${currentTabTask.stepCount}`;
    els.currentTabStatus.textContent = 'Running';
    els.currentTabStatus.className = 'tab-status status-running';
  } else {
    toggleClass(els.inputSection, 'hidden', false);
    toggleClass(els.runningSection, 'hidden', true);
    els.currentTabStatus.textContent = 'Ready';
    els.currentTabStatus.className = 'tab-status status-ready';
  }

  // Show log for selected task
  const taskToShow = tasks.find(t => t.tabId === selectedTaskTabId) || currentTabTask;
  if (taskToShow?.actionLog && taskToShow.actionLog.length !== lastLogLength) {
    // If same task, only append new entries; if different task, re-render from scratch
    // Compare both tabId AND startedAt to detect new tasks on the same tab
    const isSameTask = taskToShow.tabId === lastRenderedTaskTabId && taskToShow.startedAt === lastRenderedTaskStartedAt;
    const appendFrom = isSameTask ? lastLogLength : 0;
    renderLog(taskToShow.actionLog, appendFrom);
    lastLogLength = taskToShow.actionLog.length;
    lastRenderedTaskTabId = taskToShow.tabId;
    lastRenderedTaskStartedAt = taskToShow.startedAt;
    els.logTitle.textContent = taskToShow.tabId !== currentTabId ? `Log: Tab ${taskToShow.tabId}` : 'Action Log';
  } else if (!taskToShow?.actionLog?.length && lastLogLength !== 0) {
    renderLog([]);
    lastLogLength = 0;
    lastRenderedTaskTabId = null;
    lastRenderedTaskStartedAt = null;
    els.logTitle.textContent = 'Action Log';
  }
}

function toggleClass(el, className, condition) {
  el.classList.toggle(className, condition);
}

function renderActiveTasks(tasks) {
  els.activeTasksList.innerHTML = tasks.map(task => {
    const isCurrentTab = task.tabId === currentTabId;
    const title = task.tabTitle ? truncate(task.tabTitle, 30) : `Tab ${task.tabId}`;
    return `
      <div class="active-task-item${task.tabId === selectedTaskTabId ? ' selected' : ''}">
        <div class="task-info" data-tab-id="${task.tabId}" title="Click to go to this tab">
          <div class="task-title-row">
            <span class="task-title">${escapeHtml(title)}</span>
            ${isCurrentTab ? '<span class="current-tab-label">(current tab)</span>' : ''}
            <svg class="goto-icon" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"></path>
              <polyline points="15 3 21 3 21 9"></polyline>
              <line x1="10" y1="14" x2="21" y2="3"></line>
            </svg>
          </div>
          <div class="task-details">
            <span class="task-step">Step ${task.stepCount}</span>
            <span class="task-desc">${escapeHtml(truncate(task.currentTask || '', 40))}</span>
          </div>
        </div>
        <button class="btn btn-small btn-danger stop-task-btn" data-tab-id="${task.tabId}">Stop</button>
      </div>
    `;
  }).join('');

  // Attach event listeners
  els.activeTasksList.querySelectorAll('.task-info').forEach(el => {
    el.addEventListener('click', async () => {
      const tabId = parseInt(el.dataset.tabId);
      selectedTaskTabId = tabId;
      lastLogLength = -1;
      try {
        await chrome.tabs.update(tabId, { active: true });
        const tab = await chrome.tabs.get(tabId);
        if (tab.windowId) await chrome.windows.update(tab.windowId, { focused: true });
      } catch (e) {}
      refreshState();
    });
  });

  els.activeTasksList.querySelectorAll('.stop-task-btn').forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      stopTaskByTabId(parseInt(btn.dataset.tabId));
    });
  });
}

function renderCompletedTasks(tasks) {
  // Create a hash of task data to detect actual changes
  const expandedArray = Array.from(expandedCompletedTasks).sort();
  const tasksHash = JSON.stringify({
    tasks: tasks.map(t => ({
      tabId: t.tabId,
      status: t.completionStatus,
      answer: t.finalAnswer,
      task: t.currentTask
    })),
    expanded: expandedArray,
    currentTabId
  });

  // Skip re-render if nothing changed (preserves scroll position)
  if (tasksHash === lastCompletedTasksHash) {
    return;
  }
  lastCompletedTasksHash = tasksHash;

  // Save scroll positions before re-render
  const listScrollTop = els.completedTasksList.scrollTop;
  const answerScrollPositions = {};
  els.completedTasksList.querySelectorAll('.answer-content').forEach(el => {
    const tabId = el.closest('.completed-task-item')?.querySelector('[data-tab-id]')?.dataset.tabId;
    if (tabId && el.scrollTop > 0) {
      answerScrollPositions[tabId] = el.scrollTop;
    }
  });

  els.completedTasksList.innerHTML = tasks.map(task => {
    const isCurrentTab = task.tabId === currentTabId;
    const isExpanded = expandedCompletedTasks.has(task.tabId);
    const hasAnswer = task.finalAnswer?.trim().length > 0;
    const title = task.currentTask ? truncate(task.currentTask, 50) : 'Unknown task';
    const isWrappingUp = task.completionStatus === 'wrapping_up';
    const timeText = isWrappingUp ? 'Wrapping up...' : getTimeAgo(task.completedAt);

    return `
      <div class="completed-task-item${isCurrentTab ? ' current-tab' : ''}${isWrappingUp ? ' wrapping-up' : ''}" data-task-tab-id="${task.tabId}">
        <div class="completed-task-header" data-tab-id="${task.tabId}">
          <div class="completed-task-info">
            <span class="completed-status-icon ${getStatusClass(task.completionStatus)}">${getStatusIcon(task.completionStatus)}</span>
            <span class="completed-task-title">${escapeHtml(title)}</span>
            ${isCurrentTab ? '<span class="current-tab-badge">This tab</span>' : ''}
          </div>
          <div class="completed-task-meta">
            <span class="completed-time${isWrappingUp ? ' wrapping-up-text' : ''}">${timeText}</span>
            ${hasAnswer ? `
              <button class="expand-btn${isExpanded ? ' expanded' : ''}" data-tab-id="${task.tabId}">
                <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                  <polyline points="6 9 12 15 18 9"></polyline>
                </svg>
              </button>
            ` : ''}
            ${!isWrappingUp ? `
              <button class="dismiss-btn" data-tab-id="${task.tabId}" title="Dismiss">
                <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                  <line x1="18" y1="6" x2="6" y2="18"></line>
                  <line x1="6" y1="6" x2="18" y2="18"></line>
                </svg>
              </button>
            ` : ''}
          </div>
        </div>
        ${hasAnswer ? `
          <div class="completed-task-answer ${isExpanded ? 'expanded' : 'collapsed'}">
            <div class="answer-content">${escapeHtml(task.finalAnswer)}</div>
          </div>
        ` : ''}
      </div>
    `;
  }).join('');

  // Restore scroll positions after re-render
  els.completedTasksList.scrollTop = listScrollTop;
  Object.entries(answerScrollPositions).forEach(([tabId, scrollTop]) => {
    const answerEl = els.completedTasksList.querySelector(`[data-task-tab-id="${tabId}"] .answer-content`);
    if (answerEl) {
      answerEl.scrollTop = scrollTop;
    }
  });

  // Attach event listeners
  els.completedTasksList.querySelectorAll('.completed-task-header').forEach(el => {
    el.addEventListener('click', (e) => {
      if (!e.target.closest('.dismiss-btn')) {
        toggleExpandedTask(parseInt(el.dataset.tabId));
      }
    });
  });

  els.completedTasksList.querySelectorAll('.expand-btn').forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      toggleExpandedTask(parseInt(btn.dataset.tabId));
    });
  });

  els.completedTasksList.querySelectorAll('.dismiss-btn').forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      dismissCompletedTask(parseInt(btn.dataset.tabId));
    });
  });
}

function toggleExpandedTask(tabId) {
  expandedCompletedTasks.has(tabId) ? expandedCompletedTasks.delete(tabId) : expandedCompletedTasks.add(tabId);
  lastCompletedTasksHash = null; // Force re-render
  refreshState();
}

async function dismissCompletedTask(tabId) {
  try {
    await chrome.runtime.sendMessage({ type: 'DISMISS_COMPLETED', tabId });
    expandedCompletedTasks.delete(tabId);
    lastCompletedTasksHash = null; // Force re-render
    refreshState();
  } catch (e) {}
}

async function clearAllCompleted() {
  try {
    await chrome.runtime.sendMessage({ type: 'CLEAR_COMPLETED' });
    expandedCompletedTasks.clear();
    lastCompletedTasksHash = null; // Force re-render
    refreshState();
  } catch (e) {}
}

function getStatusIcon(status) {
  return { completed: '✓', stopped: '■', error: '✕', max_steps: '⏱', wrapping_up: '⏳' }[status] || '•';
}

function getStatusClass(status) {
  return { completed: 'status-completed', stopped: 'status-stopped', error: 'status-error', max_steps: 'status-max-steps', wrapping_up: 'status-wrapping-up' }[status] || '';
}

function getTimeAgo(timestamp) {
  if (!timestamp) return '';
  const seconds = Math.floor((Date.now() - timestamp) / 1000);
  if (seconds < 60) return 'just now';
  if (seconds < 3600) return `${Math.floor(seconds / 60)}m ago`;
  if (seconds < 86400) return `${Math.floor(seconds / 3600)}h ago`;
  return `${Math.floor(seconds / 86400)}d ago`;
}

function renderLog(log, appendFrom = 0) {
  if (!log?.length) {
    toggleClass(els.welcomeMessage, 'hidden', false);
    els.logContainer.querySelectorAll('.log-entry').forEach(e => e.remove());
    return;
  }

  // Check if user is at or near bottom before adding new entries (50px threshold)
  // Only relevant when appending; on full render we always scroll to bottom
  const wasAtBottom = appendFrom > 0
    ? els.logContainer.scrollHeight - els.logContainer.scrollTop <= els.logContainer.clientHeight + 50
    : true;

  // If appendFrom is 0, clear and render everything (e.g., switching tasks)
  if (appendFrom === 0) {
    toggleClass(els.welcomeMessage, 'hidden', true);
    els.logContainer.querySelectorAll('.log-entry').forEach(e => e.remove());
  }

  // Only render entries from appendFrom onwards
  const entriesToRender = log.slice(appendFrom);

  entriesToRender.forEach(entry => {
    const el = document.createElement('div');
    el.className = `log-entry log-${entry.type}`;

    const content = escapeHtml(entry.content);
    switch (entry.type) {
      case 'task':
        el.innerHTML = `<span class="log-label">Task:</span> ${content}`;
        break;
      case 'thoughts':
        el.innerHTML = `<span class="log-label">Thinking:</span> ${content}`;
        break;
      case 'action':
        el.innerHTML = `<span class="log-label">Action:</span> <span class="action-type">${entry.content}</span>${formatActionDetails(entry.details)}`;
        break;
      case 'status':
        el.innerHTML = `<span class="log-status">${content}</span>`;
        break;
      case 'error':
        el.innerHTML = `<span class="log-label">Error:</span> ${content}`;
        break;
      case 'complete':
        el.innerHTML = `<span class="log-label">Complete:</span> ${content}`;
        break;
      case 'system':
        el.innerHTML = `<span class="log-system">${content}</span>`;
        break;
      default:
        el.textContent = entry.content;
    }

    els.logContainer.appendChild(el);
  });

  // Only auto-scroll if user was already at the bottom
  if (wasAtBottom) {
    els.logContainer.scrollTop = els.logContainer.scrollHeight;
  }
}

function formatActionDetails(details) {
  if (!details) return '';
  const parts = [];
  const actionType = details.action_type || details.type;

  // Handle drag action: show start and end coordinates
  if (actionType === 'drag' && details.start_coordinates) {
    const endCoords = details.end_coordinates || details.center_coordinates;
    if (endCoords) {
      parts.push(`from (${details.start_coordinates[0]}, ${details.start_coordinates[1]}) to (${endCoords[0]}, ${endCoords[1]})`);
    }
  } else if (details.center_coordinates) {
    parts.push(`at (${details.center_coordinates[0]}, ${details.center_coordinates[1]})`);
  }

  if (details.text) parts.push(`"${details.text}"`);
  if (details.direction) parts.push(details.direction);
  if (details.url) parts.push(details.url);
  if (details.keys) parts.push(details.keys.join('+'));
  return parts.length ? ` <span class="action-details">${escapeHtml(parts.join(' '))}</span>` : '';
}

function truncate(str, maxLen) {
  return str.length <= maxLen ? str : str.substring(0, maxLen - 3) + '...';
}

function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

// Settings
async function loadSettings() {
  const { apiKey, apiEndpoint } = await chrome.storage.local.get(['apiKey', 'apiEndpoint']);
  if (apiKey) els.apiKeyInput.value = apiKey;
  if (apiEndpoint) els.apiEndpointInput.value = apiEndpoint;
}

async function saveSettings() {
  const apiKey = els.apiKeyInput.value.trim();
  const apiEndpoint = els.apiEndpointInput.value.trim();

  if (!apiKey) {
    alert('Please enter an API key');
    return;
  }

  await chrome.storage.local.set({ apiKey, apiEndpoint });
  hideSettings();
  checkApiKey();
}

function checkApiKey() {
  chrome.storage.local.get(['apiKey'], (result) => {
    if (!result.apiKey) showSettings();
  });
}

function showSettings() {
  toggleClass(els.settingsPanel, 'hidden', false);
  toggleClass(els.mainPanel, 'hidden', true);
}

function hideSettings() {
  toggleClass(els.settingsPanel, 'hidden', true);
  toggleClass(els.mainPanel, 'hidden', false);
}

// Task Management
async function startTask() {
  const task = els.taskInput.value.trim();
  if (!task) {
    alert('Please enter a task');
    return;
  }

  const { apiKey } = await chrome.storage.local.get(['apiKey']);
  if (!apiKey) {
    showSettings();
    return;
  }

  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab.url || tab.url.startsWith('chrome://') || tab.url.startsWith('chrome-extension://')) {
      alert('Cannot run on Chrome system pages (new tab, settings, etc.) due to browser security restrictions.\n\nPlease navigate to any website first (e.g., google.com), then start your task. The agent can navigate from there.');
      return;
    }

    const response = await chrome.runtime.sendMessage({ type: 'START_TASK', task, tabId: tab.id });
    if (response.error) {
      alert('Error: ' + response.error);
    } else {
      els.taskInput.value = '';
      selectedTaskTabId = tab.id;
    }
  } catch (error) {
    alert('Error: ' + error.message);
  }
}

async function stopTask() {
  await stopTaskByTabId(currentTabId);
}

async function stopTaskByTabId(tabId) {
  try {
    await chrome.runtime.sendMessage({ type: 'STOP_TASK', tabId });
  } catch (e) {}
}

async function clearLog() {
  try {
    await chrome.runtime.sendMessage({ type: 'CLEAR_HISTORY', tabId: selectedTaskTabId });
    lastLogLength = 0;
    renderLog([]);
  } catch (e) {}
}

// Event Listeners
els.settingsBtn.addEventListener('click', showSettings);
els.saveSettingsBtn.addEventListener('click', saveSettings);
els.cancelSettingsBtn.addEventListener('click', hideSettings);
els.startBtn.addEventListener('click', startTask);
els.stopBtn.addEventListener('click', stopTask);
els.clearBtn.addEventListener('click', clearLog);
els.clearCompletedBtn.addEventListener('click', clearAllCompleted);

els.taskInput.addEventListener('keydown', (e) => {
  if (e.key === 'Enter' && !e.shiftKey) {
    e.preventDefault();
    startTask();
  }
});
