// n1 Browser Agent - Background Service Worker

// ============ Constants ============

const DEFAULT_API_ENDPOINT = 'https://api.yutori.com/v1/chat/completions';
const MODEL_NAME = 'n1-20260122';
const DEBUG_LOGGING = false; // Set to true for verbose logging

function debugLog(...args) {
  if (DEBUG_LOGGING) console.log(...args);
}

const COORD_SCALE = 1000;
const MODEL_SCREENSHOT_WIDTH = 1280;
const MODEL_SCREENSHOT_HEIGHT = 800;
const MAX_STEPS = 75;
const MAX_HISTORY_IMAGES = 2; // Only include images for the last N turns to prevent 413 errors
const Z_INDEX_OVERLAY = 2147483646;
const Z_INDEX_TOP = 2147483647;

// Animation timing constants (in milliseconds)
// These control both visual display AND screenshot timing to ensure clean captures.
// Flow: animation starts -> LEAD_TIME delay -> action executes -> waitForPendingDelays() before next screenshot
// Delays are tracked as promises via addPendingDelay(). waitForPendingDelays() uses Promise.all to wait for all concurrent delays.
const ANIMATION_TIMING = {
  LEAD_TIME: 400,                 // Delay before action executes (lets user see animation start)
  CLICK_PULSE_DURATION: 600,      // Duration of each click pulse
  CLICK_PULSE_OVERLAP: 0.5,       // Overlap ratio between consecutive pulses (0.5 = 50% overlap)
  CLICK_CLEANUP_BUFFER: 100,      // Extra time before cleanup
  SCROLL_DURATION: 600,          // Total scroll indicator duration
  TYPE_DURATION: 600,            // Total typing indicator duration
};

// Per-action-type delays before next screenshot (in milliseconds)
// These run concurrently with animation timers - screenshot waits for max(action_delay, animation)
const ACTION_DELAYS = {
  // Click variants
  left_click: 600,
  double_click: 600,
  triple_click: 600,
  right_click: 600,
  click: 600,       // Legacy
  // Input
  type: 300,
  scroll: 300,
  key_press: 300,
  key: 300,
  hover: 300,
  drag: 300,
  // Navigation
  goto_url: 1500,
  goto: 1500,
  go_back: 1500,
  back: 1500,
  refresh: 1500,
  // Control
  wait: 0,          // No delay needed - wait action already has built-in delay
  stop: 0,
};

// URLs to block (tracking/analytics)
const BLOCKED_URL_KEYWORDS = [
  'www.facebook.com/tr',
  'connect.facebook.net',
  'googletagmanager.com',
  'google-analytics.com',
];

const TASK_INSTRUCTIONS = `

Complete the task efficiently. Once the task is complete, provide a concise summary of the results and any information you found.`;

async function getUserLocation() {
  try {
    // Try IP-based geolocation (no permission required)
    const response = await fetch('https://ipapi.co/json/', { signal: AbortSignal.timeout(3000) });
    if (response.ok) {
      const data = await response.json();
      if (data.city && data.country_name) {
        return `${data.city}, ${data.country_name}`;
      }
    }
  } catch (e) {
    // Fallback to None if geolocation fails
  }
  return null;
}

function formatTaskWithContext(task, userLocation, timezone, utcDate, utcTime, utcDayOfWeek) {
  return `# Task:
${task}
${TASK_INSTRUCTIONS}

# User Context
User's location: ${userLocation || 'None'}
User's timezone: ${timezone}

# Time Context
Current Date: ${utcDate}
Current Time: ${utcTime}
Today is: ${utcDayOfWeek}`;
}

async function computeTaskContext(task) {
  const now = new Date();
  const timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
  const userLocation = await getUserLocation();

  // Time context in UTC
  const utcDate = now.toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric', timeZone: 'UTC' });
  const utcTime = now.toLocaleTimeString('en-US', { hour: '2-digit', minute: '2-digit', timeZone: 'UTC' });
  const utcDayOfWeek = now.toLocaleDateString('en-US', { weekday: 'long', timeZone: 'UTC' });

  return formatTaskWithContext(task, userLocation, timezone, utcDate, utcTime, utcDayOfWeek);
}

// Key mappings for keyboard actions
const MODIFIER_KEYS = new Set([
  'ctrl', 'control', 'alt', 'option', 'shift', 'meta', 'cmd', 'command', 'win', 'windows'
]);

const SPECIAL_KEY_MAP = {
  'enter': { key: 'Enter', code: 'Enter', keyCode: 13 },
  'tab': { key: 'Tab', code: 'Tab', keyCode: 9 },
  'escape': { key: 'Escape', code: 'Escape', keyCode: 27 },
  'esc': { key: 'Escape', code: 'Escape', keyCode: 27 },
  'backspace': { key: 'Backspace', code: 'Backspace', keyCode: 8 },
  'delete': { key: 'Delete', code: 'Delete', keyCode: 46 },
  'arrowup': { key: 'ArrowUp', code: 'ArrowUp', keyCode: 38 },
  'arrowdown': { key: 'ArrowDown', code: 'ArrowDown', keyCode: 40 },
  'arrowleft': { key: 'ArrowLeft', code: 'ArrowLeft', keyCode: 37 },
  'arrowright': { key: 'ArrowRight', code: 'ArrowRight', keyCode: 39 },
  'up': { key: 'ArrowUp', code: 'ArrowUp', keyCode: 38 },
  'down': { key: 'ArrowDown', code: 'ArrowDown', keyCode: 40 },
  'left': { key: 'ArrowLeft', code: 'ArrowLeft', keyCode: 37 },
  'right': { key: 'ArrowRight', code: 'ArrowRight', keyCode: 39 },
  'space': { key: ' ', code: 'Space', keyCode: 32 },
  ' ': { key: ' ', code: 'Space', keyCode: 32 },
  'home': { key: 'Home', code: 'Home', keyCode: 36 },
  'end': { key: 'End', code: 'End', keyCode: 35 },
  'pageup': { key: 'PageUp', code: 'PageUp', keyCode: 33 },
  'pagedown': { key: 'PageDown', code: 'PageDown', keyCode: 34 }
};

// ============ Utilities ============

function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

function validateCoordinates(coords) {
  return coords && Array.isArray(coords) && coords.length >= 2;
}

// ============ State Management ============

const agentStates = new Map();
const completedTasksHistory = new Map(); // Per-tab history of completed tasks
const lastClickCoords = new Map(); // Track last click position per tab for type animations
const viewportCache = new Map(); // Cache viewport info per tab to avoid repeated CDP calls
const pendingDelays = new Map(); // Track pending delay promises per tab

function createAgentState(tabId, task) {
  return {
    isRunning: true,
    shouldStop: false,
    currentTask: task,
    formattedTask: null, // Computed once at task start with context
    tabId,
    tabTitle: '',
    tabUrl: '',
    conversationHistory: [],
    actionLog: [],
    stepCount: 0,
    maxSteps: MAX_STEPS,
    debuggerAttached: false,
    startedAt: Date.now(),
    completedAt: null,
    completionStatus: null,
    finalAnswer: null
  };
}

function getAllStates() {
  const activeTasks = [];
  const completedTasks = [];

  // Add completed tasks from history (per-tab)
  for (const [tabId, history] of completedTasksHistory) {
    for (const task of history) {
      completedTasks.push(task);
    }
  }

  for (const [tabId, state] of agentStates) {
    const taskInfo = {
      tabId,
      isRunning: state.isRunning,
      currentTask: state.currentTask,
      actionLog: state.actionLog,
      stepCount: state.stepCount,
      tabTitle: state.tabTitle,
      tabUrl: state.tabUrl,
      startedAt: state.startedAt,
      completedAt: state.completedAt,
      completionStatus: state.completionStatus,
      finalAnswer: state.finalAnswer
    };

    if (state.isRunning) {
      activeTasks.push(taskInfo);
    } else if (state.completedAt) {
      completedTasks.push(taskInfo);
    }
  }

  completedTasks.sort((a, b) => (a.completedAt || 0) - (b.completedAt || 0));
  return { tasks: activeTasks, completedTasks };
}

function getTabState(tabId) {
  const state = agentStates.get(tabId);
  if (!state) {
    return { isRunning: false, currentTask: null, actionLog: [], stepCount: 0 };
  }
  return {
    tabId,
    isRunning: state.isRunning,
    currentTask: state.currentTask,
    actionLog: state.actionLog,
    stepCount: state.stepCount,
    tabTitle: state.tabTitle,
    tabUrl: state.tabUrl
  };
}

function addToActionLog(tabId, type, content, details = null) {
  const state = agentStates.get(tabId);
  if (!state) return;
  state.actionLog.push({ type, content, details, timestamp: Date.now() });
  scheduleStateUpdate();
}

// Add completed task to history with deduplication
function addToCompletedHistory(tabId, taskEntry) {
  if (!completedTasksHistory.has(tabId)) {
    completedTasksHistory.set(tabId, []);
  }
  const history = completedTasksHistory.get(tabId);

  // Check for duplicate (same startedAt timestamp)
  if (history.some(t => t.startedAt === taskEntry.startedAt)) {
    return; // Already in history
  }

  // Add new entry (copy actionLog to avoid shared references)
  history.push({
    ...taskEntry,
    actionLog: [...taskEntry.actionLog]
  });
}

// Debounced state update to avoid excessive storage writes
let stateUpdatePending = false;
let stateUpdateTimer = null;

function scheduleStateUpdate() {
  if (stateUpdatePending) return; // Already scheduled
  stateUpdatePending = true;

  // Clear any existing timer
  if (stateUpdateTimer) clearTimeout(stateUpdateTimer);

  // Debounce: update after 50ms of no new calls, or max 200ms
  stateUpdateTimer = setTimeout(() => {
    stateUpdatePending = false;
    stateUpdateTimer = null;
    // Fire and forget - don't await storage write
    chrome.storage.local.set({ agentStates: getAllStates() });
    // Also broadcast to WebSocket controller if connected
    broadcastStateToController();
  }, 50);
}

function updateStoredState() {
  // Immediate update (for critical state changes like task completion)
  if (stateUpdateTimer) {
    clearTimeout(stateUpdateTimer);
    stateUpdateTimer = null;
  }
  stateUpdatePending = false;
  chrome.storage.local.set({ agentStates: getAllStates() });
  // Also broadcast to WebSocket controller (important for completion notification)
  broadcastStateToController();
}

function clearHistory(tabId) {
  if (tabId) {
    const state = agentStates.get(tabId);
    if (state && !state.isRunning) {
      agentStates.delete(tabId);
      lastClickCoords.delete(tabId);
    }
  } else {
    for (const [tid, state] of agentStates) {
      if (!state.isRunning) {
        agentStates.delete(tid);
        lastClickCoords.delete(tid);
      }
    }
  }
  updateStoredState();
}

function clearCompletedTasks() {
  for (const [tabId, state] of agentStates) {
    if (!state.isRunning && state.completedAt) {
      agentStates.delete(tabId);
      lastClickCoords.delete(tabId);
    }
  }
  updateStoredState();
}

function dismissCompletedTask(tabId) {
  const state = agentStates.get(tabId);
  if (state && !state.isRunning) {
    agentStates.delete(tabId);
    lastClickCoords.delete(tabId);
    updateStoredState();
  }
}

// ============ WebSocket Remote Control ============
// Allows external tools to control the extension programmatically.
// DISABLED by default for security. Enable via chrome.storage if needed for development.

let wsConnection = null;
let wsReconnectTimeout = null;
const WS_PORT = 9222;
const WS_REMOTE_CONTROL_ENABLED = false; // Set to true to enable remote control

function connectToController() {
  if (!WS_REMOTE_CONTROL_ENABLED || wsConnection) return;

  try {
    wsConnection = new WebSocket(`ws://localhost:${WS_PORT}`);

    wsConnection.onopen = () => {
      wsConnection.send(JSON.stringify({
        type: 'extensionReady',
        extensionId: chrome.runtime.id
      }));
    };

    wsConnection.onmessage = async (event) => {
      try {
        const msg = JSON.parse(event.data);
        await handleControllerMessage(msg);
      } catch (e) {
        // Ignore message errors
      }
    };

    wsConnection.onerror = () => {};

    wsConnection.onclose = () => {
      wsConnection = null;
      if (WS_REMOTE_CONTROL_ENABLED && !wsReconnectTimeout) {
        wsReconnectTimeout = setTimeout(() => {
          wsReconnectTimeout = null;
          connectToController();
        }, 3000);
      }
    };
  } catch (e) {
    // Server not running, ignore
  }
}

async function handleControllerMessage(msg) {
  switch (msg.type) {
    case 'startTask': {
      let tabId = msg.tabId;
      if (msg.startUrl) {
        const tab = await chrome.tabs.create({ url: msg.startUrl, active: true });
        tabId = tab.id;
        await sleep(2000);
      } else {
        const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
        tabId = tabs[0]?.id;
      }

      if (tabId) {
        sendToController({ type: 'taskStarted', tabId, task: msg.task });
        try {
          await startAgentLoop(msg.task, tabId);
        } catch (e) {
          sendToController({ type: 'taskError', tabId, error: e.message });
        }
      } else {
        sendToController({ type: 'taskError', error: 'No active tab found' });
      }
      break;
    }
    case 'stopTask':
      if (msg.tabId) {
        await stopAgentLoop(msg.tabId);
      }
      break;
    case 'ping':
      sendToController({ type: 'pong' });
      break;
  }
}

function sendToController(data) {
  if (wsConnection?.readyState === WebSocket.OPEN) {
    wsConnection.send(JSON.stringify(data));
  }
}

function broadcastStateToController() {
  if (!wsConnection || wsConnection.readyState !== WebSocket.OPEN) return;
  sendToController({ type: 'stateUpdate', ...getAllStates() });
}

// Only connect if remote control is enabled
if (WS_REMOTE_CONTROL_ENABLED) {
  connectToController();
}

// ============ Port Connection Handler (for detecting side panel close) ============

chrome.runtime.onConnect.addListener((port) => {
  if (port.name.startsWith('sidepanel-')) {
    const tabId = parseInt(port.name.replace('sidepanel-', ''), 10);
    debugLog('[N1] Side panel connected for tab:', tabId);

    port.onDisconnect.addListener(() => {
      debugLog('[N1] Side panel disconnected for tab:', tabId);
      const state = agentStates.get(tabId);
      if (state && state.isRunning) {
        debugLog('[N1] Stopping task due to panel close for tab:', tabId);
        stopAgentLoop(tabId);
      }
      tabsWithSidePanel.delete(tabId);
    });
  }
});

// ============ Message Handler ============

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case 'SIDE_PANEL_CLOSED':
      // When panel is closed, stop any running task and remove from tracking
      if (message.tabId) {
        const state = agentStates.get(message.tabId);
        if (state && state.isRunning) {
          debugLog('[N1] Panel closed - stopping task for tab:', message.tabId);
          stopAgentLoop(message.tabId);
        }
        tabsWithSidePanel.delete(message.tabId);
        debugLog('[N1] Panel closed for tab:', message.tabId);
      }
      sendResponse({ success: true });
      return true;
    case 'START_TASK':
      startAgentLoop(message.task, message.tabId)
        .then(() => sendResponse({ success: true }))
        .catch(error => sendResponse({ error: error.message }));
      return true;
    case 'STOP_TASK':
      stopAgentLoop(message.tabId);
      sendResponse({ success: true });
      return true;
    case 'GET_STATE':
      const allStates = getAllStates();
      // Log occasionally to avoid spam (every 10th call or when there are completed tasks)
      if (allStates.completedTasks?.length > 0) {
        debugLog('[N1 State] GET_STATE returning:', allStates.tasks.length, 'active,', allStates.completedTasks.length, 'completed');
        if (allStates.completedTasks.length > 0) {
          const latest = allStates.completedTasks[allStates.completedTasks.length - 1];
          debugLog('[N1 State] Latest completed:', latest.completionStatus, 'answer:', latest.finalAnswer?.substring(0, 50));
        }
      }
      sendResponse(allStates);
      return true;
    case 'GET_TAB_STATE':
      sendResponse(getTabState(message.tabId));
      return true;
    case 'CLEAR_HISTORY':
      clearHistory(message.tabId);
      sendResponse({ success: true });
      return true;
    case 'CLEAR_COMPLETED':
      clearCompletedTasks();
      sendResponse({ success: true });
      return true;
    case 'DISMISS_COMPLETED':
      dismissCompletedTask(message.tabId);
      sendResponse({ success: true });
      return true;
  }
});

// ============ Side Panel ============

debugLog('[N1] Service worker starting...');

// Track which tabs have the side panel open
const tabsWithSidePanel = new Set();

// Open side panel for a specific tab (like Claude extension pattern)
async function openSidePanelForTab(tabId) {
  debugLog('[N1] Opening side panel for tab:', tabId);

  // Set options with tabId in the path (like Claude does)
  chrome.sidePanel.setOptions({
    tabId: tabId,
    path: `sidepanel.html?tabId=${encodeURIComponent(tabId)}`,
    enabled: true
  });

  // Open the panel
  chrome.sidePanel.open({ tabId: tabId });

  // Track this tab
  tabsWithSidePanel.add(tabId);

  // Disable panel for all other tabs
  const tabs = await chrome.tabs.query({});
  for (const tab of tabs) {
    if (tab.id && tab.id !== tabId && !tabsWithSidePanel.has(tab.id)) {
      chrome.sidePanel.setOptions({ tabId: tab.id, enabled: false }).catch(() => {});
    }
  }
}

// Handle extension icon click
chrome.action.onClicked.addListener((tab) => {
  if (tab.id) {
    openSidePanelForTab(tab.id);
  }
});

// When switching tabs, show/hide panel based on tracking
chrome.tabs.onActivated.addListener(async (activeInfo) => {
  const tabId = activeInfo.tabId;

  if (tabsWithSidePanel.has(tabId)) {
    // Tracked tab - ensure panel is enabled with correct path
    chrome.sidePanel.setOptions({
      tabId: tabId,
      path: `sidepanel.html?tabId=${encodeURIComponent(tabId)}`,
      enabled: true
    }).catch(() => {});
  } else if (tabsWithSidePanel.size > 0) {
    // Untracked tab - disable panel
    chrome.sidePanel.setOptions({
      tabId: tabId,
      enabled: false
    }).catch(() => {});
  }
});

// When a new tab is created, disable panel if we're tracking any tabs
chrome.tabs.onCreated.addListener(async (tab) => {
  if (tab.id && tabsWithSidePanel.size > 0) {
    chrome.sidePanel.setOptions({
      tabId: tab.id,
      enabled: false
    }).catch(() => {});
  }
});

// Cleanup when tab is closed
chrome.tabs.onRemoved.addListener((tabId) => {
  tabsWithSidePanel.delete(tabId);
});

// ============ CDP / Debugger ============

async function attachDebugger(tabId) {
  const state = agentStates.get(tabId);
  if (!state || state.debuggerAttached) return;

  return new Promise((resolve, reject) => {
    chrome.debugger.attach({ tabId }, '1.3', async () => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
        return;
      }
      state.debuggerAttached = true;
      try {
        await sendCDP(tabId, 'Runtime.enable');
        await sendCDP(tabId, 'Page.enable');
        await sendCDP(tabId, 'DOM.enable');
        await sendCDP(tabId, 'Input.enable').catch(() => {});
        // Enable Fetch interception to block tracking URLs
        await sendCDP(tabId, 'Fetch.enable', { patterns: [{ urlPattern: '*' }] }).catch(() => {});
      } catch (e) {
        // Some domains may fail, continue anyway
      }
      resolve();
    });
  });
}

async function detachDebugger(tabId) {
  const state = agentStates.get(tabId);
  if (!state || !state.debuggerAttached) return;

  return new Promise((resolve) => {
    chrome.debugger.detach({ tabId }, () => {
      state.debuggerAttached = false;
      resolve();
    });
  });
}

async function sendCDP(tabId, method, params = {}) {
  const state = agentStates.get(tabId);
  if (!state || !state.debuggerAttached) {
    throw new Error('Debugger not attached');
  }

  return new Promise((resolve, reject) => {
    chrome.debugger.sendCommand({ tabId }, method, params, (result) => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else {
        resolve(result);
      }
    });
  });
}

// Handle CDP events (for Fetch interception)
chrome.debugger.onEvent.addListener((source, method, params) => {
  if (method === 'Fetch.requestPaused') {
    const { requestId, request } = params;
    const url = request?.url || '';
    const tabId = source.tabId;

    // Check if URL should be blocked
    const shouldBlock = BLOCKED_URL_KEYWORDS.some(keyword => url.includes(keyword));

    if (shouldBlock) {
      debugLog('[N1] Blocking URL:', url);
      chrome.debugger.sendCommand({ tabId }, 'Fetch.failRequest', {
        requestId,
        errorReason: 'BlockedByClient'
      }, () => {});
    } else {
      chrome.debugger.sendCommand({ tabId }, 'Fetch.continueRequest', { requestId }, () => {});
    }
  }
});

async function evaluateInPage(tabId, code) {
  return sendCDP(tabId, 'Runtime.evaluate', { expression: `(function() { ${code} })()` });
}

async function getViewportInfo(tabId, forceRefresh = false) {
  // Use cached viewport if available and not forcing refresh
  if (!forceRefresh && viewportCache.has(tabId)) {
    return viewportCache.get(tabId);
  }

  try {
    const { result } = await sendCDP(tabId, 'Runtime.evaluate', {
      expression: `JSON.stringify({
        width: window.innerWidth,
        height: window.innerHeight,
        devicePixelRatio: window.devicePixelRatio || 1
      })`,
      returnByValue: true
    });
    const viewport = JSON.parse(result.value);
    viewportCache.set(tabId, viewport);
    return viewport;
  } catch (e) {
    return viewportCache.get(tabId) || { width: 1280, height: 800, devicePixelRatio: 1 };
  }
}

async function convertCoordinates(tabId, coords) {
  const viewport = await getViewportInfo(tabId);
  const screenshotX = (coords[0] / COORD_SCALE) * MODEL_SCREENSHOT_WIDTH;
  const screenshotY = (coords[1] / COORD_SCALE) * MODEL_SCREENSHOT_HEIGHT;
  const x = screenshotX * (viewport.width / MODEL_SCREENSHOT_WIDTH);
  const y = screenshotY * (viewport.height / MODEL_SCREENSHOT_HEIGHT);
  return { x, y, viewport };
}

async function waitForPageLoad(tabId, maxWaitMs = 15000) {
  const startTime = Date.now();
  while (Date.now() - startTime < maxWaitMs) {
    await sleep(500);
    try {
      const { result } = await sendCDP(tabId, 'Runtime.evaluate', {
        expression: 'document.readyState',
        returnByValue: true
      });
      if (result.value === 'complete' || result.value === 'interactive') {
        await sleep(300);
        return true;
      }
    } catch (e) {
      // Page might be navigating
    }
  }
  return false;
}

// ============ Page Injections (Blocker, Indicator, Link Override) ============

// Cleanup function that runs in page context - used by scripting API
// This is the single source of truth for page cleanup logic
function pageCleanupFunction() {
  document.getElementById('__n1BlockerOverlay')?.remove();
  const indicator = document.getElementById('__n1BlockedIndicator');
  if (indicator) {
    if (indicator.__animationId) cancelAnimationFrame(indicator.__animationId);
    indicator.remove();
  }

  if (window.__n1BlockerHandler) {
    const events = window.__n1BlockerEvents || ['keydown', 'keyup', 'keypress'];
    events.forEach(evt => {
      document.removeEventListener(evt, window.__n1BlockerHandler, window.__n1BlockerOpts);
      window.removeEventListener(evt, window.__n1BlockerHandler, window.__n1BlockerOpts);
    });
  }
  window.__n1BlockerActive = false;
  delete window.__n1BlockerHandler;
  delete window.__n1BlockerOpts;
  delete window.__n1BlockerEvents;

  if (window.__n1OpenOverridden && window.__n1OriginalOpen) {
    window.open = window.__n1OriginalOpen;
    delete window.__n1OriginalOpen;
    delete window.__n1OpenOverridden;
  }

  if (window.__n1OriginalSetAttribute) {
    Element.prototype.setAttribute = window.__n1OriginalSetAttribute;
    delete window.__n1OriginalSetAttribute;
    delete Element.prototype._n1SetAttributePatched;
  }

  if (window.__n1OriginalFormTargetDescriptor) {
    Object.defineProperty(HTMLFormElement.prototype, 'target', window.__n1OriginalFormTargetDescriptor);
    delete window.__n1OriginalFormTargetDescriptor;
    delete HTMLFormElement.prototype._n1TargetPatched;
  }

  if (window.__n1OriginalAnchorTargetDescriptor) {
    Object.defineProperty(HTMLAnchorElement.prototype, 'target', window.__n1OriginalAnchorTargetDescriptor);
    delete window.__n1OriginalAnchorTargetDescriptor;
    delete HTMLAnchorElement.prototype._n1TargetPatched;
  }

  if (window.__n1SubmitHandler) {
    document.removeEventListener('submit', window.__n1SubmitHandler, true);
    delete window.__n1SubmitHandler;
    delete window._n1SubmitListenerPatched;
  }

  if (window.__n1LinkObserver) {
    window.__n1LinkObserver.disconnect();
    delete window.__n1LinkObserver;
  }
}

// String version for CDP evaluation (derived from the function above)
const PAGE_CLEANUP_SCRIPT = `(${pageCleanupFunction.toString()})();`;

async function injectInteractionBlocker(tabId) {
  const state = agentStates.get(tabId);
  if (!state || !state.isRunning) return;

  try {
    await evaluateInPage(tabId, `
      // Create or show overlay
      let overlay = document.getElementById('__n1BlockerOverlay');
      if (overlay) {
        overlay.style.opacity = '1';
        overlay.style.pointerEvents = 'auto';
      } else {
        overlay = document.createElement('div');
        overlay.id = '__n1BlockerOverlay';
        overlay.style.cssText = 'position:fixed;top:0;left:0;width:100vw;height:100vh;z-index:${Z_INDEX_OVERLAY};background:transparent;cursor:not-allowed;opacity:1;transition:opacity 0.05s ease-out;';

        const blockEvent = e => { e.preventDefault(); e.stopPropagation(); e.stopImmediatePropagation(); return false; };
        const blockedEvents = ['mousedown','mouseup','click','dblclick','contextmenu','wheel','mousemove','mouseover','mouseenter','mouseleave','mouseout','touchstart','touchmove','touchend','pointerdown','pointerup','pointermove','pointerover','pointerenter','pointerleave','pointerout','scroll'];
        blockedEvents.forEach(evt => overlay.addEventListener(evt, blockEvent, { capture: true, passive: false }));

        document.documentElement.appendChild(overlay);
      }

      // Block keyboard events
      const keyEvents = ['keydown', 'keyup', 'keypress'];
      if (window.__n1BlockerHandler) {
        keyEvents.forEach(evt => {
          document.removeEventListener(evt, window.__n1BlockerHandler, window.__n1BlockerOpts);
          window.removeEventListener(evt, window.__n1BlockerHandler, window.__n1BlockerOpts);
        });
      }
      const handler = e => { e.preventDefault(); e.stopPropagation(); e.stopImmediatePropagation(); return false; };
      const opts = { capture: true, passive: false };
      keyEvents.forEach(evt => {
        document.addEventListener(evt, handler, opts);
        window.addEventListener(evt, handler, opts);
      });
      window.__n1BlockerActive = true;
      window.__n1BlockerHandler = handler;
      window.__n1BlockerOpts = opts;
      window.__n1BlockerEvents = keyEvents;
    `);
    await forceLinksInSameTab(tabId);
  } catch (e) {}
}

async function removeInteractionBlocker(tabId) {
  // Try CDP first (faster if debugger attached)
  try {
    await evaluateInPage(tabId, PAGE_CLEANUP_SCRIPT);
    return;
  } catch (e) {}

  // Fallback to scripting API (works without debugger)
  try {
    await chrome.scripting.executeScript({
      target: { tabId },
      func: pageCleanupFunction
    });
  } catch (e) {}
}

async function suspendInteractionBlocker(tabId) {
  const state = agentStates.get(tabId);
  if (!state || !state.isRunning) return;

  try {
    await evaluateInPage(tabId, `
      const overlay = document.getElementById('__n1BlockerOverlay');
      if (overlay) {
        overlay.style.opacity = '0';
        overlay.style.pointerEvents = 'none';
      }

      if (window.__n1BlockerHandler) {
        const events = window.__n1BlockerEvents || [];
        events.forEach(evt => {
          document.removeEventListener(evt, window.__n1BlockerHandler, window.__n1BlockerOpts);
          window.removeEventListener(evt, window.__n1BlockerHandler, window.__n1BlockerOpts);
        });
      }
    `);
  } catch (e) {}
}

async function showBlockedIndicator(tabId) {
  const state = agentStates.get(tabId);
  if (!state || !state.isRunning) return;

  try {
    await evaluateInPage(tabId, `
      let indicator = document.getElementById('__n1BlockedIndicator');
      if (indicator) {
        indicator.style.opacity = '1';
        return;
      }

      indicator = document.createElement('div');
      indicator.id = '__n1BlockedIndicator';
      indicator.style.cssText = 'position:fixed;top:0;left:0;right:0;bottom:0;pointer-events:none;z-index:${Z_INDEX_TOP};opacity:1;transition:opacity 0.3s ease-out;';

      const gradientBorder = document.createElement('div');
      gradientBorder.id = '__n1GradientBorder';
      gradientBorder.style.cssText = 'position:absolute;inset:0;pointer-events:none;filter:blur(8px);';

      indicator.appendChild(gradientBorder);
      document.documentElement.appendChild(indicator);

      // Animate gradient using requestAnimationFrame (like Framer Motion)
      const duration = 4000; // 4 seconds like landing page
      let startTime = null;
      let animationId = null;

      // Easing function: easeInOut
      function easeInOut(t) {
        return t < 0.5 ? 2 * t * t : 1 - Math.pow(-2 * t + 2, 2) / 2;
      }

      // Interpolate between two values
      function lerp(a, b, t) {
        return a + (b - a) * t;
      }

      function animate(timestamp) {
        if (!startTime) startTime = timestamp;
        const elapsed = timestamp - startTime;
        const cycle = (elapsed % duration) / duration; // 0 to 1

        // Create ping-pong effect (0->1->0)
        const pingPong = cycle < 0.5 ? cycle * 2 : 2 - cycle * 2;
        const t = easeInOut(pingPong);

        // Interpolate values (matching landing page)
        const opacity = lerp(0.25, 0.4, t);
        const spread = lerp(8, 15, t);

        const gradient = \`
          linear-gradient(to right, rgba(29, 205, 152, \${opacity}) 0%, transparent \${spread}%),
          linear-gradient(to left, rgba(29, 205, 152, \${opacity}) 0%, transparent \${spread}%),
          linear-gradient(to bottom, rgba(29, 205, 152, \${opacity}) 0%, transparent \${spread}%),
          linear-gradient(to top, rgba(29, 205, 152, \${opacity}) 0%, transparent \${spread}%)
        \`;

        const el = document.getElementById('__n1GradientBorder');
        if (el) {
          el.style.background = gradient;
          animationId = requestAnimationFrame(animate);
        }
      }

      animationId = requestAnimationFrame(animate);

      // Store animation ID for cleanup
      indicator.__animationId = animationId;
    `);
  } catch (e) {}
}

async function hideBlockedIndicator(tabId) {
  try {
    await evaluateInPage(tabId, `
      const indicator = document.getElementById('__n1BlockedIndicator');
      if (indicator) {
        // Cancel the animation
        if (indicator.__animationId) {
          cancelAnimationFrame(indicator.__animationId);
        }
        indicator.style.opacity = '0';
      }
    `);
  } catch (e) {}
}

async function forceLinksInSameTab(tabId) {
  try {
    await evaluateInPage(tabId, `
      // 1. Remove target from all elements that could open new windows
      const removeTargets = () => {
        document.querySelectorAll('[target], [formtarget]').forEach(el => {
          const target = el.getAttribute('target') || el.getAttribute('formtarget');
          if (target && target !== '_self' && target !== '_parent' && target !== '_top') {
            el.removeAttribute('target');
            el.removeAttribute('formtarget');
          }
        });
      };
      removeTargets();

      // 2. Override window.open - save original for cleanup restoration
      if (!window.__n1OpenOverridden) {
        window.__n1OriginalOpen = window.open;
        window.__n1OpenOverridden = true;
        Object.defineProperty(window, 'open', {
          value: function(url, name, specs) {
            if (typeof url === 'string' && url && !url.startsWith('about:')) {
              window.location.href = url;
            }
            return { closed: false, focus: () => {}, blur: () => {}, close: () => {}, postMessage: () => {} };
          },
          writable: true,
          configurable: true
        });
      }

      // 3. Intercept setAttribute to catch element.setAttribute('target', ...)
      if (!Element.prototype._n1SetAttributePatched) {
        window.__n1OriginalSetAttribute = Element.prototype.setAttribute;
        Element.prototype.setAttribute = function(name, value) {
          if ((name.toLowerCase() === 'target' || name.toLowerCase() === 'formtarget') &&
              value && value !== '_self' && value !== '_parent' && value !== '_top') {
            return; // Block it
          }
          return window.__n1OriginalSetAttribute.call(this, name, value);
        };
        Element.prototype._n1SetAttributePatched = true;
      }

      // 4. Prevent form.target from being set to new-tab values
      if (!HTMLFormElement.prototype._n1TargetPatched) {
        window.__n1OriginalFormTargetDescriptor = Object.getOwnPropertyDescriptor(HTMLFormElement.prototype, 'target');
        Object.defineProperty(HTMLFormElement.prototype, 'target', {
          set: function(val) {
            if (!val || val === '_self' || val === '_parent' || val === '_top') {
              this.setAttribute('target', val || '');
            }
          },
          get: function() { return this.getAttribute('target') || ''; },
          configurable: true
        });
        HTMLFormElement.prototype._n1TargetPatched = true;
      }

      // 5. Prevent anchor.target from being set to new-tab values
      if (!HTMLAnchorElement.prototype._n1TargetPatched) {
        window.__n1OriginalAnchorTargetDescriptor = Object.getOwnPropertyDescriptor(HTMLAnchorElement.prototype, 'target');
        Object.defineProperty(HTMLAnchorElement.prototype, 'target', {
          set: function(val) {
            if (!val || val === '_self' || val === '_parent' || val === '_top') {
              this.setAttribute('target', val || '');
            }
          },
          get: function() { return this.getAttribute('target') || ''; },
          configurable: true
        });
        HTMLAnchorElement.prototype._n1TargetPatched = true;
      }

      // 6. Monitor form submissions to ensure bad targets are removed
      if (!window._n1SubmitListenerPatched) {
        window.__n1SubmitHandler = (e) => {
          const target = e.target.getAttribute('target');
          if (target && target !== '_self' && target !== '_parent' && target !== '_top') {
            e.target.removeAttribute('target');
          }
        };
        document.addEventListener('submit', window.__n1SubmitHandler, true);
        window._n1SubmitListenerPatched = true;
      }

      // 7. Watch for new elements with target attributes
      if (!window.__n1LinkObserver) {
        window.__n1LinkObserver = new MutationObserver(removeTargets);
        window.__n1LinkObserver.observe(document.documentElement, {
          childList: true, subtree: true, attributes: true, attributeFilter: ['target', 'formtarget']
        });
      }
    `);
  } catch (e) {}
}

// ============ Visual Effects ============

// Add a delay promise that must complete before next screenshot
function addPendingDelay(tabId, durationMs) {
  if (!pendingDelays.has(tabId)) {
    pendingDelays.set(tabId, []);
  }
  pendingDelays.get(tabId).push(sleep(durationMs));
}

// Wait for all pending delays to complete (animation timers, post-action delays, etc.)
async function waitForPendingDelays(tabId) {
  const delays = pendingDelays.get(tabId);
  if (delays && delays.length > 0) {
    await Promise.all(delays);
    pendingDelays.delete(tabId);
  }
}

async function showClickEffect(tabId, x, y, numClicks = 1) {
  const px = Math.round(x);
  const py = Math.round(y);
  const { CLICK_PULSE_DURATION, CLICK_PULSE_OVERLAP, CLICK_CLEANUP_BUFFER } = ANIMATION_TIMING;
  const pulseGap = CLICK_PULSE_DURATION * CLICK_PULSE_OVERLAP;
  const totalDuration = (numClicks - 1) * pulseGap + CLICK_PULSE_DURATION + CLICK_CLEANUP_BUFFER;
  addPendingDelay(tabId, totalDuration);
  try {
    await evaluateInPage(tabId, `
      if (!document.getElementById('__n1ClickStyle')) {
        const style = document.createElement('style');
        style.id = '__n1ClickStyle';
        style.textContent = '@keyframes n1click{0%{transform:translate(-50%,-50%) scale(0.4);opacity:1}50%{opacity:0.9}100%{transform:translate(-50%,-50%) scale(2.5);opacity:0}}@keyframes n1clickdot{0%{transform:translate(-50%,-50%) scale(1);opacity:1}50%{opacity:1}100%{transform:translate(-50%,-50%) scale(0.4);opacity:0}}';
        document.head.appendChild(style);
      }

      const numClicks = ${numClicks};
      const pulseDuration = ${CLICK_PULSE_DURATION};
      const pulseGap = ${pulseGap};
      const cleanupBuffer = ${CLICK_CLEANUP_BUFFER};
      for (let i = 0; i < numClicks; i++) {
        const delay = i * pulseGap;

        const container = document.createElement('div');
        container.className = '__n1ClickPulse';
        container.style.cssText = 'position:fixed;left:${px}px;top:${py}px;pointer-events:none;z-index:${Z_INDEX_OVERLAY};';

        // Center dot - bright solid green
        const dot = document.createElement('div');
        dot.style.cssText = 'position:absolute;left:0;top:0;width:14px;height:14px;background:#1DCD98;border-radius:50%;transform:translate(-50%,-50%);animation:n1clickdot ' + (pulseDuration/1000) + 's ease-out forwards;box-shadow:0 0 16px #1DCD98, 0 0 32px rgba(29,205,152,0.5);animation-delay:' + delay + 'ms;opacity:0;';

        // Outer ring - bright green with strong glow
        const ring = document.createElement('div');
        ring.style.cssText = 'position:absolute;left:0;top:0;width:50px;height:50px;border:3px solid #1DCD98;border-radius:50%;transform:translate(-50%,-50%);animation:n1click ' + (pulseDuration/1000) + 's ease-out forwards;box-shadow:0 0 20px rgba(29,205,152,0.6), inset 0 0 12px rgba(29,205,152,0.2);animation-delay:' + delay + 'ms;opacity:0;';

        container.appendChild(ring);
        container.appendChild(dot);
        document.body.appendChild(container);

        setTimeout(() => container.remove(), delay + pulseDuration + cleanupBuffer);
      }
    `);
  } catch (e) {}
}

async function showScrollEffect(tabId, x, y, direction) {
  const px = Math.round(x);
  const py = Math.round(y);
  const duration = ANIMATION_TIMING.SCROLL_DURATION;
  addPendingDelay(tabId, duration);
  const rotation = { up: -90, down: 90, left: 180, right: 0 }[direction] || 0;
  const moveX = direction === 'left' ? -10 : (direction === 'right' ? 10 : 0);
  const moveY = direction === 'up' ? -10 : (direction === 'down' ? 10 : 0);

  try {
    await evaluateInPage(tabId, `
      const oldStyle = document.getElementById('__n1ScrollStyle');
      if (oldStyle) oldStyle.remove();

      const duration = ${duration};
      const style = document.createElement('style');
      style.id = '__n1ScrollStyle';
      style.textContent = \`
        @keyframes n1scrollchevron {
          0%, 100% { transform: translate(0, 0); opacity: 0.6; }
          50% { transform: translate(${moveX}px, ${moveY}px); opacity: 1; }
        }
        @keyframes n1scrollfade {
          0% { opacity: 0; transform: translate(-50%, -50%) scale(0.9); }
          15% { opacity: 1; transform: translate(-50%, -50%) scale(1); }
          85% { opacity: 1; }
          100% { opacity: 0; transform: translate(-50%, -50%) scale(0.95); }
        }
      \`;
      document.head.appendChild(style);

      const container = document.createElement('div');
      container.style.cssText = 'position:fixed;left:${px}px;top:${py}px;pointer-events:none;z-index:${Z_INDEX_OVERLAY};transform:translate(-50%,-50%);animation:n1scrollfade ' + (duration/1000) + 's ease-out forwards;';

      const box = document.createElement('div');
      box.style.cssText = 'width:56px;height:56px;border:2.5px solid #1DCD98;border-radius:12px;display:flex;align-items:center;justify-content:center;box-shadow:0 0 20px rgba(29,205,152,0.4);';

      const chevron = document.createElement('div');
      chevron.innerHTML = '<svg width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="#1DCD98" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round" style="transform:rotate(${rotation}deg)"><polyline points="9 18 15 12 9 6"></polyline></svg>';
      chevron.style.cssText = 'animation:n1scrollchevron 0.7s ease-in-out infinite;display:flex;';

      box.appendChild(chevron);
      container.appendChild(box);
      document.body.appendChild(container);
      setTimeout(() => { container.remove(); style.remove(); }, duration);
    `);
  } catch (e) {}
}

async function showTypeEffect(tabId, x, y) {
  const px = Math.round(x);
  // Position above by default, but below if too close to top of screen
  // Indicator is ~30px tall, so if y < 50, it would be cut off at top
  const showBelow = y < 50;
  const py = showBelow ? Math.round(y) + 30 : Math.round(y) - 36;
  const duration = ANIMATION_TIMING.TYPE_DURATION;
  addPendingDelay(tabId, duration);
  try {
    await evaluateInPage(tabId, `
      const oldStyle = document.getElementById('__n1TypeStyle');
      if (oldStyle) oldStyle.remove();

      const showBelow = ${showBelow};
      const duration = ${duration};
      const style = document.createElement('style');
      style.id = '__n1TypeStyle';
      style.textContent = \`
        @keyframes n1typebob {
          0%, 100% { transform: translateX(-50%) translateY(0); }
          50% { transform: translateX(-50%) translateY(\${showBelow ? '6px' : '-6px'}); }
        }
        @keyframes n1typeglow {
          0%, 100% { box-shadow: 0 4px 16px rgba(29,205,152,0.5); }
          50% { box-shadow: 0 4px 28px rgba(29,205,152,0.9), 0 0 12px rgba(29,205,152,0.5); }
        }
        @keyframes n1typedot {
          0%, 100% { opacity: 0.2; transform: translateY(0) scale(0.8); }
          50% { opacity: 1; transform: translateY(\${showBelow ? '5px' : '-5px'}) scale(1.1); }
        }
        @keyframes n1typecursor {
          0%, 100% { opacity: 1; }
          50% { opacity: 0; }
        }
        @keyframes n1typefade {
          0% { opacity: 0; transform: translateX(-50%) translateY(\${showBelow ? '-10px' : '10px'}); }
          15% { opacity: 1; transform: translateX(-50%) translateY(0); }
          85% { opacity: 1; }
          100% { opacity: 0; transform: translateX(-50%) translateY(\${showBelow ? '10px' : '-10px'}); }
        }
      \`;
      document.head.appendChild(style);

      const indicator = document.createElement('div');
      indicator.style.cssText = 'position:fixed;left:${px}px;top:${py}px;background:#1DCD98;color:#000;font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;font-size:11px;font-weight:600;letter-spacing:0.5px;text-transform:uppercase;padding:6px 12px;border-radius:5px;pointer-events:none;z-index:${Z_INDEX_OVERLAY};transform:translateX(-50%);animation:n1typefade ' + (duration/1000) + 's ease-out forwards,n1typebob 0.6s ease-in-out infinite,n1typeglow 0.8s ease-in-out infinite;box-shadow:0 4px 16px rgba(29,205,152,0.5);display:flex;align-items:center;gap:3px;';

      const text = document.createElement('span');
      text.textContent = 'typing';
      indicator.appendChild(text);

      for (let i = 0; i < 3; i++) {
        const dot = document.createElement('span');
        dot.textContent = '·';
        dot.style.cssText = 'display:inline-block;font-size:14px;font-weight:700;line-height:1;animation:n1typedot 0.5s ease-in-out infinite;animation-delay:' + (i * 0.1) + 's;';
        indicator.appendChild(dot);
      }

      const cursor = document.createElement('span');
      cursor.style.cssText = 'width:2px;height:12px;background:#000;margin-left:4px;border-radius:1px;animation:n1typecursor 0.53s step-end infinite;';
      indicator.appendChild(cursor);

      document.body.appendChild(indicator);
      setTimeout(() => { indicator.remove(); style.remove(); }, duration);
    `);
  } catch (e) {}
}

// ============ CDP Actions ============

async function cdpClick(tabId, action) {
  // Support both API format (coordinates) and legacy format (center_coordinates)
  const coords = action.coordinates || action.center_coordinates;
  if (!validateCoordinates(coords)) {
    return { skipped: true, reason: 'Missing coordinates' };
  }

  const { x, y } = await convertCoordinates(tabId, coords);
  const numClicks = action.num_clicks || action.click_count || 1;
  const button = action.button === 'right' ? 'right' : 'left';
  const clickX = Math.round(x);
  const clickY = Math.round(y);

  // Store click coordinates for subsequent type actions
  lastClickCoords.set(tabId, { x: clickX, y: clickY });

  await sendCDP(tabId, 'Input.dispatchMouseEvent', { type: 'mouseMoved', x: clickX, y: clickY });
  await sleep(20);

  // Show animation once before all clicks (with multiple pulses for multi-click)
  showClickEffect(tabId, clickX, clickY, numClicks); // Fire-and-forget animation
  await sleep(ANIMATION_TIMING.LEAD_TIME);

  // Execute clicks rapidly (within dblclick timing window)
  for (let i = 0; i < numClicks; i++) {
    await sendCDP(tabId, 'Input.dispatchMouseEvent', {
      type: 'mousePressed', x: clickX, y: clickY, button, clickCount: i + 1, buttons: 1
    });
    await sleep(15);
    await sendCDP(tabId, 'Input.dispatchMouseEvent', {
      type: 'mouseReleased', x: clickX, y: clickY, button, clickCount: i + 1, buttons: 0
    });
    if (i < numClicks - 1) await sleep(50); // Short gap between clicks
  }

  return { clicked: true, x: clickX, y: clickY };
}

async function cdpType(tabId, action) {
  const text = action.text || '';

  // Support both API format (coordinates) and legacy format (center_coordinates)
  const coords = action.coordinates || action.center_coordinates;
  if (coords) {
    await cdpClick(tabId, { coordinates: coords });
    await sleep(50);
    const { x, y } = await convertCoordinates(tabId, coords);
    showTypeEffect(tabId, Math.round(x), Math.round(y)); // Fire-and-forget
    await sleep(ANIMATION_TIMING.LEAD_TIME);
  } else if (lastClickCoords.has(tabId)) {
    // Use coordinates from the previous click action
    const { x, y } = lastClickCoords.get(tabId);
    showTypeEffect(tabId, x, y); // Fire-and-forget
    await sleep(ANIMATION_TIMING.LEAD_TIME);
  }

  // Support both API format (clear_before_typing) and legacy format (clear_before_type)
  if (action.clear_before_typing || action.clear_before_type) {
    // Select all using JavaScript (platform-independent), then delete
    await evaluateInPage(tabId, `
      const el = document.activeElement;
      if (el && el.select) {
        el.select();
      } else {
        document.execCommand('selectAll');
      }
    `);
    await sleep(15);
    await sendCDP(tabId, 'Input.dispatchKeyEvent', { type: 'keyDown', key: 'Backspace', code: 'Backspace' });
    await sendCDP(tabId, 'Input.dispatchKeyEvent', { type: 'keyUp', key: 'Backspace', code: 'Backspace' });
    await sleep(20);
  }

  await sendCDP(tabId, 'Input.insertText', { text });

  // Support both API format (press_enter_after) and legacy format (press_enter)
  if (action.press_enter_after || action.press_enter) {
    await sleep(20);
    await sendCDP(tabId, 'Input.dispatchKeyEvent', {
      type: 'keyDown', key: 'Enter', code: 'Enter', windowsVirtualKeyCode: 13, nativeVirtualKeyCode: 13
    });
    await sendCDP(tabId, 'Input.dispatchKeyEvent', { type: 'keyUp', key: 'Enter', code: 'Enter' });
  }

  return { typed: true, text };
}

async function cdpScroll(tabId, action) {
  const direction = action.direction || 'down';
  const rawAmount = action.amount || 3;
  const amount = rawAmount < 50 ? rawAmount * 300 : rawAmount;

  let x, y;
  // Support both API format (coordinates) and legacy format (center_coordinates)
  const coords = action.coordinates || action.coordinate || action.center_coordinates;
  if (coords) {
    const converted = await convertCoordinates(tabId, coords);
    x = converted.x;
    y = converted.y;
  } else {
    const viewport = await getViewportInfo(tabId);
    x = viewport.width / 2;
    y = viewport.height / 2;
  }

  showScrollEffect(tabId, Math.round(x), Math.round(y), direction); // Fire-and-forget
  await sleep(ANIMATION_TIMING.LEAD_TIME);

  const deltaMap = { up: [0, -amount], down: [0, amount], left: [-amount, 0], right: [amount, 0] };
  const [deltaX, deltaY] = deltaMap[direction] || [0, amount];

  await sendCDP(tabId, 'Input.dispatchMouseEvent', { type: 'mouseWheel', x, y, deltaX, deltaY });
  return { scrolled: true, direction, amount };
}

async function cdpKeyPress(tabId, action) {
  // Support both API format (key_comb string) and legacy format (keys array)
  let keys = action.keys || [];
  if (keys.length === 0 && action.key_comb) {
    keys = action.key_comb.split('+').map(k => k.trim());
  }
  // Also support single 'key' field (alternative API format)
  if (keys.length === 0 && action.key) {
    keys = [action.key];
  }
  if (keys.length === 0) return { keyPressed: false, reason: 'No keys specified' };

  // Separate modifiers from regular keys
  const modifierKeys = [];
  const regularKeys = [];
  for (const key of keys) {
    if (MODIFIER_KEYS.has(key.toLowerCase())) {
      modifierKeys.push(key);
    } else {
      regularKeys.push(key);
    }
  }

  // Handle Ctrl/Cmd + clipboard shortcuts using JavaScript (CDP doesn't reliably trigger these)
  const commandModifiers = new Set(['ctrl', 'control', 'meta', 'cmd', 'command']);
  const hasCommandModifier = modifierKeys.some(m => commandModifiers.has(m.toLowerCase()));

  if (hasCommandModifier && regularKeys.length === 1) {
    const regularKey = regularKeys[0].toLowerCase();

    // Select All: Ctrl+A / Cmd+A
    if (regularKey === 'a') {
      await evaluateInPage(tabId, `
        const el = document.activeElement;
        if (el && el.select) {
          el.select();
        } else {
          document.execCommand('selectAll');
        }
      `);
      return { keyPressed: true, keys, action: 'select-all' };
    }

    // Copy: Ctrl+C / Cmd+C
    if (regularKey === 'c') {
      await evaluateInPage(tabId, `document.execCommand('copy')`);
      return { keyPressed: true, keys, action: 'copy' };
    }

    // Paste: Ctrl+V / Cmd+V
    if (regularKey === 'v') {
      await evaluateInPage(tabId, `document.execCommand('paste')`);
      return { keyPressed: true, keys, action: 'paste' };
    }

    // Cut: Ctrl+X / Cmd+X
    if (regularKey === 'x') {
      await evaluateInPage(tabId, `document.execCommand('cut')`);
      return { keyPressed: true, keys, action: 'cut' };
    }

    // Undo: Ctrl+Z / Cmd+Z
    if (regularKey === 'z') {
      await evaluateInPage(tabId, `document.execCommand('undo')`);
      return { keyPressed: true, keys, action: 'undo' };
    }

    // Redo: Ctrl+Y / Cmd+Y
    if (regularKey === 'y') {
      await evaluateInPage(tabId, `document.execCommand('redo')`);
      return { keyPressed: true, keys, action: 'redo' };
    }
  }

  // For any modifier+key combinations, use CDP with modifiers bitmask
  if (modifierKeys.length > 0 && regularKeys.length === 1) {
    const regularKey = regularKeys[0].toLowerCase();

    // Calculate modifiers: Alt=1, Ctrl=2, Meta/Command=4, Shift=8
    let modifiers = 0;
    for (const mod of modifierKeys) {
      const m = mod.toLowerCase();
      if (m === 'alt' || m === 'option') modifiers |= 1;
      if (m === 'ctrl' || m === 'control') modifiers |= 2;
      if (m === 'meta' || m === 'cmd' || m === 'command' || m === 'win' || m === 'windows') modifiers |= 4;
      if (m === 'shift') modifiers |= 8;
    }

    const specialInfo = SPECIAL_KEY_MAP[regularKey];
    const info = specialInfo || {
      key: regularKeys[0].length === 1 ? regularKeys[0] : regularKeys[0],
      code: regularKeys[0].length === 1 ? `Key${regularKeys[0].toUpperCase()}` : regularKeys[0],
      keyCode: regularKeys[0].length === 1 ? regularKeys[0].toUpperCase().charCodeAt(0) : 0
    };

    await sendCDP(tabId, 'Input.dispatchKeyEvent', {
      type: 'rawKeyDown',
      key: info.key,
      code: info.code,
      windowsVirtualKeyCode: info.keyCode,
      modifiers
    });
    await sleep(20);

    await sendCDP(tabId, 'Input.dispatchKeyEvent', {
      type: 'keyUp',
      key: info.key,
      code: info.code,
      windowsVirtualKeyCode: info.keyCode,
      modifiers
    });

    return { keyPressed: true, keys, modifiers };
  }

  // For any modifier+key combinations, use CDP with modifiers bitmask
  if (modifierKeys.length > 0 && regularKeys.length === 1) {
    const regularKey = regularKeys[0].toLowerCase();

    // Calculate modifiers: Alt=1, Ctrl=2, Meta/Command=4, Shift=8
    let modifiers = 0;
    for (const mod of modifierKeys) {
      const m = mod.toLowerCase();
      if (m === 'alt' || m === 'option') modifiers |= 1;
      if (m === 'ctrl' || m === 'control') modifiers |= 2;
      if (m === 'meta' || m === 'cmd' || m === 'command' || m === 'win' || m === 'windows') modifiers |= 4;
      if (m === 'shift') modifiers |= 8;
    }

    const specialInfo = SPECIAL_KEY_MAP[regularKey];
    const info = specialInfo || {
      key: regularKeys[0].length === 1 ? regularKeys[0] : regularKeys[0],
      code: regularKeys[0].length === 1 ? `Key${regularKeys[0].toUpperCase()}` : regularKeys[0],
      keyCode: regularKeys[0].length === 1 ? regularKeys[0].toUpperCase().charCodeAt(0) : 0
    };

    await sendCDP(tabId, 'Input.dispatchKeyEvent', {
      type: 'rawKeyDown',
      key: info.key,
      code: info.code,
      windowsVirtualKeyCode: info.keyCode,
      modifiers
    });
    await sleep(20);

    await sendCDP(tabId, 'Input.dispatchKeyEvent', {
      type: 'keyUp',
      key: info.key,
      code: info.code,
      windowsVirtualKeyCode: info.keyCode,
      modifiers
    });

    return { keyPressed: true, keys, modifiers };
  }

  // For standalone keys (Enter, Escape, Tab, arrows, etc.) use CDP
  for (const key of keys) {
    const normalizedKey = key.toLowerCase();

    // Skip modifier keys (only relevant if modifierKeys array is empty)
    if (MODIFIER_KEYS.has(normalizedKey)) continue;

    const specialInfo = SPECIAL_KEY_MAP[normalizedKey];
    const info = specialInfo || {
      key: key.length === 1 ? key : key,
      code: key.length === 1 ? `Key${key.toUpperCase()}` : key,
      keyCode: key.length === 1 ? key.toUpperCase().charCodeAt(0) : 0
    };

    const textValue = normalizedKey === 'enter' ? '\r' :
                      normalizedKey === 'tab' ? '\t' :
                      (normalizedKey === 'space' || key === ' ') ? ' ' :
                      key.length === 1 ? key : '';

    await sendCDP(tabId, 'Input.dispatchKeyEvent', {
      type: 'rawKeyDown',
      key: info.key,
      code: info.code,
      windowsVirtualKeyCode: info.keyCode,
      text: textValue,
      unmodifiedText: textValue
    });
    await sleep(10);

    if (textValue) {
      await sendCDP(tabId, 'Input.dispatchKeyEvent', {
        type: 'char',
        key: info.key,
        code: info.code,
        windowsVirtualKeyCode: info.keyCode,
        text: textValue,
        unmodifiedText: textValue
      });
      await sleep(10);
    }

    await sendCDP(tabId, 'Input.dispatchKeyEvent', {
      type: 'keyUp',
      key: info.key,
      code: info.code,
      windowsVirtualKeyCode: info.keyCode
    });
    await sleep(15);
  }

  return { keyPressed: true, keys };
}

async function cdpHover(tabId, action) {
  // Support both API format (coordinates) and legacy format (center_coordinates)
  const coords = action.coordinates || action.center_coordinates;
  if (!validateCoordinates(coords)) {
    return { skipped: true, reason: 'Missing coordinates' };
  }

  const { x, y } = await convertCoordinates(tabId, coords);
  const hoverX = Math.round(x);
  const hoverY = Math.round(y);

  await sendCDP(tabId, 'Input.dispatchMouseEvent', { type: 'mouseMoved', x: hoverX, y: hoverY });
  return { hovered: true, x: hoverX, y: hoverY };
}

async function cdpDrag(tabId, action) {
  // Support both API format and legacy format for start/end coordinates
  const startCoords = action.start_coordinates || action.startCoordinates;
  // API uses 'coordinates' for end position, legacy uses 'end_coordinates'
  const endCoords = action.coordinates || action.end_coordinates || action.endCoordinates;

  if (!validateCoordinates(startCoords)) {
    return { skipped: true, reason: 'Missing start_coordinates' };
  }
  if (!validateCoordinates(endCoords)) {
    return { skipped: true, reason: 'Missing end coordinates' };
  }

  const start = await convertCoordinates(tabId, startCoords);
  const end = await convertCoordinates(tabId, endCoords);
  const startX = Math.round(start.x);
  const startY = Math.round(start.y);
  const endX = Math.round(end.x);
  const endY = Math.round(end.y);

  debugLog(`[N1 Drag] From (${startX}, ${startY}) to (${endX}, ${endY})`);

  // Mouse down at start
  showClickEffect(tabId, startX, startY); // Fire-and-forget
  await sendCDP(tabId, 'Input.dispatchMouseEvent', {
    type: 'mousePressed', x: startX, y: startY, button: 'left', clickCount: 1
  });
  await sleep(25);

  // Move to end
  await sendCDP(tabId, 'Input.dispatchMouseEvent', {
    type: 'mouseMoved', x: endX, y: endY
  });
  await sleep(25);

  // Mouse up at end
  showClickEffect(tabId, endX, endY); // Fire-and-forget
  await sendCDP(tabId, 'Input.dispatchMouseEvent', {
    type: 'mouseReleased', x: endX, y: endY, button: 'left', clickCount: 1
  });

  return { dragged: true, start: { x: startX, y: startY }, end: { x: endX, y: endY } };
}

async function cdpNavigate(tabId, url) {
  await sendCDP(tabId, 'Page.navigate', { url });
  await waitForPageLoad(tabId);
  viewportCache.delete(tabId); // Clear viewport cache after navigation
  await forceLinksInSameTab(tabId);
  return { navigated: true, url };
}

async function cdpGoBack(tabId) {
  const history = await sendCDP(tabId, 'Page.getNavigationHistory');
  if (history.currentIndex > 0) {
    const entry = history.entries[history.currentIndex - 1];
    await sendCDP(tabId, 'Page.navigateToHistoryEntry', { entryId: entry.id });
    await waitForPageLoad(tabId);
    viewportCache.delete(tabId); // Clear viewport cache after navigation
    await forceLinksInSameTab(tabId);
  }
  return { wentBack: true };
}

async function cdpRefresh(tabId) {
  await sendCDP(tabId, 'Page.reload');
  await waitForPageLoad(tabId);
  viewportCache.delete(tabId); // Clear viewport cache after refresh
  await forceLinksInSameTab(tabId);
  return { refreshed: true };
}

// ============ Action Executor ============

async function executeAction(tabId, action) {
  const actionType = action.action_type || action.type || action.name;
  const state = agentStates.get(tabId);

  // Only verify connection if debugger not attached (skip in happy path)
  if (state && !state.debuggerAttached) {
    // Wait up to 3 seconds for reattachment
    for (let i = 0; i < 6; i++) {
      await sleep(500);
      if (state.debuggerAttached) break;
      if (state.shouldStop) throw new Error('Task stopped');
    }
    if (!state.debuggerAttached) {
      throw new Error('Lost connection to tab: Debugger not attached');
    }
  }

  // Suspend blocker during action execution (fire-and-forget for speed)
  suspendInteractionBlocker(tabId);

  let result;
  try {
    switch (actionType) {
      // Click actions - API uses left_click, double_click, triple_click, right_click
      case 'left_click':
        result = await cdpClick(tabId, { ...action, num_clicks: 1, button: 'left' });
        break;
      case 'double_click':
        result = await cdpClick(tabId, { ...action, num_clicks: 2, button: 'left' });
        break;
      case 'triple_click':
        result = await cdpClick(tabId, { ...action, num_clicks: 3, button: 'left' });
        break;
      case 'right_click':
        result = await cdpClick(tabId, { ...action, num_clicks: 1, button: 'right' });
        break;
      case 'click':
        // Legacy format support
        result = await cdpClick(tabId, action);
        break;

      case 'type':
        result = await cdpType(tabId, action);
        break;

      case 'scroll':
        result = await cdpScroll(tabId, action);
        break;

      // Key press - API uses key_press with key_comb, or key with single key
      case 'key_press':
      case 'key':
        result = await cdpKeyPress(tabId, action);
        break;

      case 'hover':
        result = await cdpHover(tabId, action);
        break;

      case 'drag':
        result = await cdpDrag(tabId, action);
        break;

      // Navigation - API uses goto_url or goto
      case 'goto_url':
      case 'goto':
        result = await cdpNavigate(tabId, action.url);
        break;

      // Back - API uses go_back or back
      case 'go_back':
      case 'back':
        result = await cdpGoBack(tabId);
        break;

      case 'refresh':
        result = await cdpRefresh(tabId);
        break;

      case 'wait':
        await sleep(action.duration || 500);
        result = { waited: true };
        break;

      case 'stop':
        result = { stopped: true, answer: action.answer };
        break;

      default:
        result = { skipped: true, reason: `Unknown action type: ${actionType}` };
    }
  } finally {
    // Re-enable blocker after action (don't wait - runs in background)
    // These are non-critical UI updates that shouldn't block the action flow
    Promise.all([injectInteractionBlocker(tabId), showBlockedIndicator(tabId)]).catch(() => {});
  }

  return result;
}

// ============ Agent Loop ============

async function startAgentLoop(task, tabId) {
  const existingState = agentStates.get(tabId);
  if (existingState && existingState.isRunning) {
    throw new Error('A task is already running on this tab');
  }

  // Save previous completed task to history before starting new one
  if (existingState && existingState.completedAt) {
    addToCompletedHistory(tabId, {
      tabId,
      isRunning: false,
      currentTask: existingState.currentTask,
      actionLog: existingState.actionLog,
      stepCount: existingState.stepCount,
      tabTitle: existingState.tabTitle,
      tabUrl: existingState.tabUrl,
      startedAt: existingState.startedAt,
      completedAt: existingState.completedAt,
      completionStatus: existingState.completionStatus,
      finalAnswer: existingState.finalAnswer
    });
  }

  const { apiKey, apiEndpoint } = await chrome.storage.local.get(['apiKey', 'apiEndpoint']);
  if (!apiKey) throw new Error('API key not configured');

  // Get tab info
  let tabTitle = '', tabUrl = '';
  try {
    const tab = await chrome.tabs.get(tabId);
    tabTitle = tab.title || '';
    tabUrl = tab.url || '';
  } catch (e) {}

  // Initialize state
  const state = createAgentState(tabId, task);
  state.tabTitle = tabTitle;
  state.tabUrl = tabUrl;

  // Compute task context once at start (includes user location and time context)
  state.formattedTask = await computeTaskContext(task);

  agentStates.set(tabId, state);

  addToActionLog(tabId, 'task', task);
  addToActionLog(tabId, 'system', `Task bound to tab ${tabId}`);
  updateStoredState();

  try {
    await attachDebugger(tabId);
    await injectInteractionBlocker(tabId);
    await showBlockedIndicator(tabId);
    await runAgentLoop(apiEndpoint || DEFAULT_API_ENDPOINT, apiKey, tabId);
  } catch (error) {
    addToActionLog(tabId, 'error', error.message);
    state.completedAt = Date.now();
    state.completionStatus = 'error';
  } finally {
    await hideBlockedIndicator(tabId);
    await removeInteractionBlocker(tabId);
    await detachDebugger(tabId);
    const finalState = agentStates.get(tabId);
    if (finalState) {
      finalState.isRunning = false;
      // Clear conversation history to free memory (screenshots are large)
      // The actionLog and finalAnswer retain all necessary information
      finalState.conversationHistory = [];
    }
    updateStoredState();
  }
}

async function stopAgentLoop(tabId) {
  debugLog('[N1 Stop] stopAgentLoop called for tab', tabId);
  const state = agentStates.get(tabId);
  if (!state) {
    debugLog('[N1 Stop] No state found, returning');
    return;
  }

  state.shouldStop = true;
  state.isRunning = false;
  state.completionStatus = 'wrapping_up';
  state.completedAt = Date.now();
  updateStoredState();
  debugLog('[N1 Stop] Set status to wrapping_up');

  addToActionLog(tabId, 'system', 'Stopping task, requesting summary...');

  // Clean up UI elements (fire-and-forget to avoid hanging on CDP issues)
  debugLog('[N1 Stop] Cleaning up UI elements (fire-and-forget)');
  hideBlockedIndicator(tabId).catch(() => {});
  removeInteractionBlocker(tabId).catch(() => {});

  // Request final summary from API (with timeout protection)
  debugLog('[N1 Stop] Calling requestFinalSummary...');
  await requestFinalSummary(tabId, state);
  debugLog('[N1 Stop] requestFinalSummary completed');

  state.completedAt = Date.now();
  state.completionStatus = 'stopped';
  state.isRunning = false;
  debugLog('[N1 Stop] Set status to stopped');

  // Detach debugger (fire-and-forget)
  detachDebugger(tabId).catch(() => {});

  // Clear conversation history to free memory (screenshots are large)
  // The actionLog and finalAnswer retain all necessary information
  state.conversationHistory = [];

  updateStoredState();
  debugLog('[N1 Stop] stopAgentLoop finished');
}

async function requestFinalSummary(tabId, state) {
  const SUMMARY_TIMEOUT_MS = 30000; // 30 second timeout for summary
  debugLog('[N1 Stop] requestFinalSummary started, timeout:', SUMMARY_TIMEOUT_MS, 'ms');

  // Wrap entire function with timeout to prevent hanging
  let timeoutId;
  const timeoutPromise = new Promise((_, reject) => {
    timeoutId = setTimeout(() => {
      debugLog('[N1 Stop] Summary timeout triggered after', SUMMARY_TIMEOUT_MS, 'ms');
      reject(new Error('Summary request timed out'));
    }, SUMMARY_TIMEOUT_MS);
  });

  try {
    debugLog('[N1 Stop] Calling requestFinalSummaryInternal...');
    await Promise.race([requestFinalSummaryInternal(tabId, state), timeoutPromise]);
    debugLog('[N1 Stop] requestFinalSummaryInternal completed successfully');
  } catch (e) {
    console.error('[N1 Stop] Summary error:', e.message);
    if (!state.finalAnswer) {
      state.finalAnswer = 'Task stopped.';
      addToActionLog(tabId, 'complete', state.finalAnswer);
    }
  } finally {
    clearTimeout(timeoutId);
    debugLog('[N1 Stop] requestFinalSummary finished, finalAnswer:', state.finalAnswer?.substring(0, 100));
  }
}

async function requestFinalSummaryInternal(tabId, state) {
  debugLog('[N1 Stop] requestFinalSummaryInternal started');

  debugLog('[N1 Stop] Getting API key from storage...');
  const { apiKey, apiEndpoint } = await chrome.storage.local.get(['apiKey', 'apiEndpoint']);
  debugLog('[N1 Stop] API key present:', !!apiKey, 'history length:', state.conversationHistory.length);

  if (!apiKey || state.conversationHistory.length === 0) {
    debugLog('[N1 Stop] No API key or empty history, setting simple stop message');
    state.finalAnswer = 'Task stopped.';
    addToActionLog(tabId, 'complete', state.finalAnswer);
    return;
  }

  const endpoint = apiEndpoint || DEFAULT_API_ENDPOINT;
  debugLog('[N1 Stop] Using endpoint:', endpoint);

  // Build messages from history without a new screenshot (uses last N existing screenshots)
  debugLog('[N1 Stop] Building messages...');
  const messages = buildMessages(state.formattedTask, null, state.conversationHistory);
  debugLog('[N1 Stop] Built', messages.length, 'messages');

  // Request summary (use original task text for clarity)
  messages.push({
    role: 'user',
    content: [{ type: 'text', text: `Stop here. Summarize your current progress and list in detail all the findings relevant to the given task: ${state.currentTask}` }]
  });

  addToActionLog(tabId, 'status', 'Getting final summary...');

  let response = null;
  try {
    debugLog('[N1 Stop] Calling API for summary...');
    const startTime = Date.now();
    response = await callN1Api(endpoint, apiKey, messages);
    debugLog('[N1 Stop] API call completed in', Date.now() - startTime, 'ms');
  } catch (e) {
    console.error('[N1 Stop] Summary API error:', e.message);
    state.finalAnswer = 'Task stopped.';
    addToActionLog(tabId, 'complete', state.finalAnswer);
    return;
  }

  if (!response) {
    debugLog('[N1 Stop] No response from API');
    state.finalAnswer = 'Task stopped.';
    addToActionLog(tabId, 'complete', state.finalAnswer);
    return;
  }

  debugLog('[N1 Stop] Response received, actions:', response.actions?.length, 'thoughts:', !!response.thoughts);

  // Extract answer from stop action or thoughts
  let finalAnswer = null;
  if (response.actions?.length > 0) {
    const stopAction = response.actions.find(a => (a.action_type || a.type) === 'stop');
    if (stopAction?.answer) finalAnswer = stopAction.answer;
    debugLog('[N1 Stop] Found stop action with answer:', !!finalAnswer);
  }
  if (!finalAnswer && response.thoughts) {
    finalAnswer = response.thoughts;
    debugLog('[N1 Stop] Using thoughts as answer');
  }

  if (finalAnswer) {
    state.finalAnswer = finalAnswer;
    addToActionLog(tabId, 'complete', finalAnswer);
    debugLog('[N1 Stop] Set final answer, length:', finalAnswer.length);
  } else {
    state.finalAnswer = 'Task stopped.';
    addToActionLog(tabId, 'complete', state.finalAnswer);
    debugLog('[N1 Stop] No answer found, using default');
  }
}

const SUPPORTED_ACTIONS = new Set([
  // Click variants
  'left_click', 'double_click', 'triple_click', 'right_click', 'click',
  // Other actions
  'type', 'scroll', 'key_press', 'key', 'hover', 'drag',
  // Navigation
  'goto_url', 'goto', 'go_back', 'back', 'refresh',
  // Control
  'wait', 'stop'
]);

// Retry configuration for API calls
const API_MAX_RETRIES = 3;
const API_RETRY_DELAY_MS = 1000;

function isActionSupported(action) {
  const actionType = action.action_type || action.type;
  return SUPPORTED_ACTIONS.has(actionType);
}

function hasUnsupportedAction(actions) {
  return actions.some(a => !isActionSupported(a));
}

async function runAgentLoop(endpoint, apiKey, tabId) {
  const state = agentStates.get(tabId);
  if (!state) return;

  while (!state.shouldStop && state.stepCount < state.maxSteps) {
    state.stepCount++;

    try {
      // Capture screenshot and get current tab URL in parallel
      addToActionLog(tabId, 'status', `Step ${state.stepCount}: Capturing screenshot...`);
      const [screenshot, currentUrl] = await Promise.all([
        captureCleanScreenshot(tabId),
        chrome.tabs.get(tabId).then(tab => tab.url || null).catch(() => null)
      ]);

      // Call API with retry logic
      addToActionLog(tabId, 'status', `Step ${state.stepCount}: Thinking...`);
      const messages = buildMessages(state.formattedTask, screenshot, state.conversationHistory, currentUrl);

      let response = null;
      let apiError = null;
      for (let attempt = 1; attempt <= API_MAX_RETRIES; attempt++) {
        try {
          response = await callN1Api(endpoint, apiKey, messages);
          apiError = null; // Clear any previous error on success
          break;
        } catch (e) {
          apiError = e;
          // For fatal errors (API key issues), don't retry
          if (e.message.includes('API key')) {
            break;
          }
          // Silent retry for transient errors (504, network issues, etc.)
          if (attempt < API_MAX_RETRIES) {
            debugLog(`[N1] API call failed (attempt ${attempt}/${API_MAX_RETRIES}), retrying...`, e.message);
            await sleep(API_RETRY_DELAY_MS * attempt);
          }
        }
      }

      // If all retries failed, throw the error to be handled below
      if (apiError) {
        throw apiError;
      }

      // Check for unsupported actions - quietly retry once
      if (response.actions && hasUnsupportedAction(response.actions)) {
        debugLog('[N1] Unsupported action detected, retrying API call...');
        response = await callN1Api(endpoint, apiKey, messages);

        // If still unsupported after retry, add to history with observation and continue
        if (response.actions && hasUnsupportedAction(response.actions)) {
          const unsupportedAction = response.actions.find(a => !isActionSupported(a));
          const actionType = unsupportedAction.action_type || unsupportedAction.type;
          debugLog(`[N1] Unsupported action "${actionType}" after retry, adding observation`);

          // Add to conversation history with observation (not surfaced to user)
          state.conversationHistory.push({
            screenshot,
            url: currentUrl,
            response,
            observation: `The action "${actionType}" is not supported. Please use a different action.`
          });
          continue; // Skip to next iteration to get new actions
        }
      }

      state.conversationHistory.push({ screenshot, url: currentUrl, response });

      // Log full model response for debugging
      debugLog('[N1 Model Response] Step', state.stepCount, {
        thoughts: response.thoughts,
        actions: response.actions
      });

      if (response.thoughts) {
        addToActionLog(tabId, 'thoughts', response.thoughts);
      }

      if (!response.actions || response.actions.length === 0) {
        // No tool_calls returned = task complete. The thoughts contain the final summary.
        const finalAnswer = response.thoughts || 'Task completed';
        debugLog('[N1 Complete] No tool_calls returned, using thoughts as final answer:', finalAnswer.substring(0, 100));
        addToActionLog(tabId, 'complete', finalAnswer);
        state.finalAnswer = finalAnswer;
        state.completedAt = Date.now();
        state.completionStatus = 'completed';
        state.isRunning = false;
        updateStoredState();

        // Clean up UI elements
        removeInteractionBlocker(tabId).catch(() => {});
        return;
      }

      // Execute actions
      for (const action of response.actions) {
        if (state.shouldStop) {
          addToActionLog(tabId, 'system', 'Stopped by user');
          return;
        }

        const actionType = action.action_type || action.type;

        if (actionType === 'stop') {
          const answer = action.answer || 'Task completed';
          debugLog('[N1 Stop Action] Task completed naturally, answer:', answer.substring(0, 100));
          addToActionLog(tabId, 'complete', answer, action);
          state.completedAt = Date.now();
          state.completionStatus = 'completed';
          state.finalAnswer = answer;
          updateStoredState();
          return;
        }

        addToActionLog(tabId, 'action', actionType, action);
        debugLog('[N1 Action] Executing:', actionType, JSON.stringify(action, null, 2));

        try {
          const result = await executeAction(tabId, action);
          debugLog('[N1 Action] Result:', JSON.stringify(result, null, 2));
          // Set per-action-type delay before next screenshot (concurrent with any animation timers)
          const actionDelay = ACTION_DELAYS[actionType] ?? 500;
          if (actionDelay > 0) {
            addPendingDelay(tabId, actionDelay);
          }
        } catch (err) {
          console.error('[N1 Action] Error:', err.message);
          addToActionLog(tabId, 'error', `Action failed: ${err.message}`);
        }
      }
      // No fixed delay needed - animation timing is tracked and waited for before next screenshot
    } catch (error) {
      // Only fatal errors (API key, screenshot failures) should be logged and rethrown
      // Transient API errors (504, etc.) have already been retried silently above
      if (error.message.includes('API key') || error.message.includes('Failed to capture')) {
        addToActionLog(tabId, 'error', `Step ${state.stepCount} error: ${error.message}`);
        throw error;
      }
      // For API errors that persisted after retries, log once and continue
      if (error.message.includes('API request failed')) {
        addToActionLog(tabId, 'error', `Step ${state.stepCount}: API error after ${API_MAX_RETRIES} retries, continuing...`);
      } else {
        addToActionLog(tabId, 'error', `Step ${state.stepCount} error: ${error.message}`);
      }
      await sleep(500);
    }
  }

  // Handle completion
  if (state.stepCount >= state.maxSteps) {
    addToActionLog(tabId, 'system', `Reached maximum steps (${state.maxSteps}), requesting summary...`);
    state.isRunning = false;
    state.completionStatus = 'wrapping_up';
    state.completedAt = Date.now();
    updateStoredState();

    // Clean up UI elements
    removeInteractionBlocker(tabId).catch(() => {});

    // Request final summary (like stop does)
    await requestFinalSummary(tabId, state);

    state.completedAt = Date.now();
    state.completionStatus = 'max_steps';

    // Add to completed tasks history
    addToCompletedHistory(tabId, {
      tabId,
      isRunning: false,
      currentTask: state.currentTask,
      actionLog: state.actionLog,
      stepCount: state.stepCount,
      tabTitle: state.tabTitle,
      tabUrl: state.tabUrl,
      startedAt: state.startedAt,
      completedAt: state.completedAt,
      completionStatus: state.completionStatus,
      finalAnswer: state.finalAnswer
    });

    // Detach debugger
    detachDebugger(tabId).catch(() => {});
    updateStoredState();
    return;
  }

  if (!state.completedAt && !state.shouldStop) {
    state.completedAt = Date.now();
    state.completionStatus = 'completed';
    updateStoredState();
  }
}

// ============ Screenshot Capture ============

async function hideOverlaysForScreenshot(tabId) {
  try {
    await evaluateInPage(tabId, `
      const indicator = document.getElementById('__n1BlockedIndicator');
      if (indicator) {
        indicator.style.transition = 'none';
        indicator.style.opacity = '0';
      }
    `);
  } catch (e) {}
}

async function showOverlaysAfterScreenshot(tabId) {
  // Recreate indicator if it was removed by the site's JavaScript
  await showBlockedIndicator(tabId);
}

async function captureScreenshot(tabId) {
  const state = agentStates.get(tabId);

  // If debugger not attached, wait briefly for reattachment
  if (state && !state.debuggerAttached && state.isRunning) {
    for (let i = 0; i < 6; i++) {
      await sleep(500);
      if (state.debuggerAttached) break;
      if (state.shouldStop) throw new Error('Task stopped');
    }
  }

  try {
    let dataUrl;
    if (state && state.debuggerAttached) {
      const result = await sendCDP(tabId, 'Page.captureScreenshot', { format: 'webp', quality: 85, captureBeyondViewport: false });
      if (!result || !result.data) throw new Error('CDP screenshot returned no data');
      dataUrl = 'data:image/webp;base64,' + result.data;
    } else {
      // Fallback to visible tab capture (works even when tab not focused)
      dataUrl = await chrome.tabs.captureVisibleTab(null, { format: 'webp', quality: 85 });
    }
    return await resizeScreenshot(dataUrl, MODEL_SCREENSHOT_WIDTH, MODEL_SCREENSHOT_HEIGHT);
  } catch (error) {
    // Try fallback method if CDP fails
    try {
      const dataUrl = await chrome.tabs.captureVisibleTab(null, { format: 'webp', quality: 85 });
      return await resizeScreenshot(dataUrl, MODEL_SCREENSHOT_WIDTH, MODEL_SCREENSHOT_HEIGHT);
    } catch (fallbackError) {
      throw new Error('Failed to capture screenshot. Make sure the tab is accessible.');
    }
  }
}

async function captureCleanScreenshot(tabId) {
  // Wait for animations AND post-action delay (whichever ends later)
  await waitForPendingDelays(tabId);
  await hideOverlaysForScreenshot(tabId);
  try {
    return await captureScreenshot(tabId);
  } finally {
    showOverlaysAfterScreenshot(tabId); // Fire-and-forget (don't wait)
  }
}

async function resizeScreenshot(dataUrl, targetWidth, targetHeight) {
  // Decode base64 directly instead of using fetch (faster)
  const base64 = dataUrl.split(',')[1];
  const mimeType = dataUrl.split(',')[0].match(/:(.*?);/)?.[1] || 'image/webp';
  const binaryStr = atob(base64);
  const bytes = new Uint8Array(binaryStr.length);
  for (let i = 0; i < binaryStr.length; i++) bytes[i] = binaryStr.charCodeAt(i);
  const blob = new Blob([bytes], { type: mimeType });

  const imageBitmap = await createImageBitmap(blob);
  const canvas = new OffscreenCanvas(targetWidth, targetHeight);
  const ctx = canvas.getContext('2d');
  ctx.drawImage(imageBitmap, 0, 0, targetWidth, targetHeight);

  const resizedBlob = await canvas.convertToBlob({ type: 'image/webp', quality: 0.85 });
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onloadend = () => resolve(reader.result);
    reader.onerror = reject;
    reader.readAsDataURL(resizedBlob);
  });
}

// ============ API Communication ============

function buildMessages(formattedTask, screenshot, history, currentUrl = null) {
  // formattedTask already includes task instructions and context (computed once at task start)
  const messages = [];

  // Include all history but only add images for the last N turns to prevent 413 errors
  // If adding a new screenshot, reserve one slot for it
  const maxHistoryImages = screenshot ? MAX_HISTORY_IMAGES - 1 : MAX_HISTORY_IMAGES;
  const imageStartIndex = Math.max(0, history.length - maxHistoryImages);

  // First user message with task
  if (history.length > 0 && imageStartIndex === 0) {
    // Include first screenshot in user message
    const firstTurn = history[0];
    messages.push({
      role: 'user',
      content: [
        { type: 'text', text: formattedTask },
        { type: 'image_url', image_url: { url: firstTurn.screenshot } }
      ]
    });

    // Add assistant response with tool_calls for first turn
    if (firstTurn.response?.toolCalls?.length > 0) {
      messages.push({
        role: 'assistant',
        content: firstTurn.response.thoughts || '',
        tool_calls: firstTurn.response.toolCalls
      });
    }

    // Add subsequent turns
    for (let i = 1; i < history.length; i++) {
      const turn = history[i];
      const prevTurn = history[i - 1];
      const includeImage = i >= imageStartIndex;

      // Add tool result with screenshot
      if (prevTurn.response?.toolCalls?.length > 0 && includeImage) {
        const toolContent = [];
        if (turn.url) {
          toolContent.push({ type: 'text', text: `Current URL: ${turn.url}` });
        }
        toolContent.push({ type: 'image_url', image_url: { url: turn.screenshot } });
        messages.push({
          role: 'tool',
          tool_call_id: prevTurn.response.toolCalls[0].id,
          content: toolContent
        });
      }

      // Add assistant response
      if (turn.response?.toolCalls?.length > 0) {
        messages.push({
          role: 'assistant',
          content: turn.response.thoughts || '',
          tool_calls: turn.response.toolCalls
        });
      }

      // Include observation text if present (e.g., for unsupported actions)
      if (turn.observation) {
        messages.push({
          role: 'user',
          content: [{ type: 'text', text: turn.observation }]
        });
      }
    }
  } else if (history.length === 0) {
    // No history yet - just add user message with task
    messages.push({ role: 'user', content: [{ type: 'text', text: formattedTask }] });
  } else {
    // imageStartIndex > 0, skipping some history images
    messages.push({ role: 'user', content: [{ type: 'text', text: formattedTask }] });

    for (let i = 0; i < history.length; i++) {
      const turn = history[i];
      const prevTurn = i > 0 ? history[i - 1] : null;
      const includeImage = i >= imageStartIndex;

      if (i === 0 && includeImage) {
        // First turn screenshot as separate user message
        messages.push({
          role: 'user',
          content: [{ type: 'image_url', image_url: { url: turn.screenshot } }]
        });
      } else if (prevTurn?.response?.toolCalls?.length > 0 && includeImage) {
        // Tool result with screenshot
        const toolContent = [];
        if (turn.url) {
          toolContent.push({ type: 'text', text: `Current URL: ${turn.url}` });
        }
        toolContent.push({ type: 'image_url', image_url: { url: turn.screenshot } });
        messages.push({
          role: 'tool',
          tool_call_id: prevTurn.response.toolCalls[0].id,
          content: toolContent
        });
      }

      // Add assistant response
      if (turn.response?.toolCalls?.length > 0) {
        messages.push({
          role: 'assistant',
          content: turn.response.thoughts || '',
          tool_calls: turn.response.toolCalls
        });
      }

      if (turn.observation) {
        messages.push({
          role: 'user',
          content: [{ type: 'text', text: turn.observation }]
        });
      }
    }
  }

  // Add current screenshot as tool result
  if (screenshot && history.length > 0) {
    const lastTurn = history[history.length - 1];
    if (lastTurn.response?.toolCalls?.length > 0) {
      const toolContent = [];
      if (currentUrl) {
        toolContent.push({ type: 'text', text: `Current URL: ${currentUrl}` });
      }
      toolContent.push({ type: 'image_url', image_url: { url: screenshot } });
      messages.push({
        role: 'tool',
        tool_call_id: lastTurn.response.toolCalls[0].id,
        content: toolContent
      });
    }
  } else if (screenshot && history.length === 0) {
    // First screenshot - add to initial user message
    const lastUserMsg = messages[messages.length - 1];
    if (lastUserMsg?.role === 'user') {
      if (Array.isArray(lastUserMsg.content)) {
        lastUserMsg.content.push({ type: 'image_url', image_url: { url: screenshot } });
      } else {
        lastUserMsg.content = [
          { type: 'text', text: lastUserMsg.content },
          { type: 'image_url', image_url: { url: screenshot } }
        ];
      }
    }
  }

  return messages;
}

const API_TIMEOUT_MS = 120000; // 2 minute timeout for API calls

async function callN1Api(endpoint, apiKey, messages) {
  // Build request body - n1-20260122 uses tool_calls natively, no response_format needed
  const requestBody = JSON.stringify({
    model: MODEL_NAME,
    messages,
    temperature: 0.3
  });

  // Debug logging for API requests
  if (DEBUG_LOGGING) {
    const msgSummary = messages.map(m => m.role + (m.content?.[0]?.image_url ? '[img]' : '')).join(', ');
    debugLog(`[N1 API] Request: ${messages.length} messages (${msgSummary})`);
  }

  // Use AbortController for timeout (covers entire request including response body)
  const controller = new AbortController();
  const timeoutId = setTimeout(() => controller.abort(), API_TIMEOUT_MS);

  try {
    const response = await fetch(endpoint, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${apiKey}`,
        'X-Client-Source': 'browser-extension',
      },
      body: requestBody,
      signal: controller.signal
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error('[N1 API] Error:', response.status, errorText);
      throw new Error(`API request failed: ${response.status}`);
    }

    const rawResponse = await response.json();
    debugLog('[N1 API] Response received');

    return parseApiResponse(rawResponse);
  } catch (e) {
    if (e.name === 'AbortError') {
      throw new Error('API request timed out');
    }
    throw e;
  } finally {
    clearTimeout(timeoutId);
  }
}

function parseApiResponse(data) {
  if (!data.choices || data.choices.length === 0) {
    return { thoughts: 'Unable to parse response', actions: [], toolCalls: [], raw: data };
  }

  const choice = data.choices[0];
  const message = choice.message;

  // n1-20260122 format: content contains reasoning, tool_calls contains actions
  const thoughts = message?.content || '';

  // Handle tool_calls-based response (primary format for n1-20260122)
  if (message?.tool_calls && message.tool_calls.length > 0) {
    const toolCalls = message.tool_calls;
    // Parse tool calls directly - use API format as-is
    const actions = toolCalls.map(tc => {
      try {
        const args = JSON.parse(tc.function.arguments);
        // Use the tool name directly as action type, merge in arguments
        return { action_type: tc.function.name, name: tc.function.name, ...args };
      } catch {
        return { action_type: 'unknown', name: tc.function.name, raw: tc.function.arguments };
      }
    });
    debugLog('[N1 Parse] Parsed tool_calls:', toolCalls.length, 'actions');
    return { thoughts, actions, toolCalls };
  }

  // Fallback: try to extract from content if no tool_calls (e.g., tool_choice="none")
  if (thoughts && typeof thoughts === 'string') {
    // Check for <tool_call> tags in content
    const toolCallMatch = thoughts.match(/<tool_call>\s*([\s\S]*?)\s*<\/tool_call>/);
    if (toolCallMatch) {
      try {
        const toolCallData = JSON.parse(toolCallMatch[1]);
        const args = toolCallData.arguments || {};
        const action = { action_type: toolCallData.name, name: toolCallData.name, ...args };
        const toolCalls = [{
          id: `call_extracted_0`,
          type: 'function',
          function: {
            name: toolCallData.name,
            arguments: JSON.stringify(args)
          }
        }];
        debugLog('[N1 Parse] Extracted tool_call from content tags');
        return { thoughts: thoughts.replace(/<tool_call>[\s\S]*?<\/tool_call>/, '').trim(), actions: [action], toolCalls };
      } catch (e) {
        debugLog('[N1 Parse] Failed to parse tool_call from content');
      }
    }
  }

  // No actions found
  debugLog('[N1 Parse] No tool_calls found, returning thoughts only');
  return { thoughts, actions: [], toolCalls: [] };
}

// ============ Event Handlers ============

chrome.debugger.onDetach.addListener(async (source, reason) => {
  const tabId = source.tabId;
  const state = agentStates.get(tabId);
  if (!state) return;

  state.debuggerAttached = false;

  // Only try to reattach if detach was NOT intentional
  // Do NOT reconnect if: tab closed, user canceled, or DevTools opened
  const shouldReconnect = state.isRunning &&
    reason !== 'target_closed' &&
    reason !== 'canceled_by_user' &&
    reason !== 'replaced_with_devtools';

  if (shouldReconnect) {
    addToActionLog(tabId, 'system', `Debugger detached (${reason}), attempting to reconnect...`);

    // Try to reattach with retries
    const maxRetries = 3;
    const retryDelay = 500;

    for (let attempt = 1; attempt <= maxRetries; attempt++) {
      await sleep(retryDelay * attempt);

      // Check if task was stopped by user during retry
      if (state.shouldStop || !state.isRunning) {
        break;
      }

      try {
        // Check if tab still exists
        await chrome.tabs.get(tabId);

        // Try to reattach debugger
        await new Promise((resolve, reject) => {
          chrome.debugger.attach({ tabId }, '1.3', () => {
            if (chrome.runtime.lastError) {
              reject(new Error(chrome.runtime.lastError.message));
            } else {
              resolve();
            }
          });
        });

        state.debuggerAttached = true;

        // Re-enable CDP domains
        try {
          await sendCDP(tabId, 'Runtime.enable');
          await sendCDP(tabId, 'Page.enable');
          await sendCDP(tabId, 'DOM.enable');
          await sendCDP(tabId, 'Input.enable').catch(() => {});
          await sendCDP(tabId, 'Fetch.enable', { patterns: [{ urlPattern: '*' }] }).catch(() => {});
        } catch (e) {
          // Some domains may fail, continue anyway
        }

        // Re-inject interaction blocker
        await injectInteractionBlocker(tabId);
        await showBlockedIndicator(tabId);

        addToActionLog(tabId, 'system', 'Reconnected to tab successfully');
        updateStoredState();
        return; // Successfully reattached
      } catch (err) {
        debugLog(`[N1] Reattach attempt ${attempt}/${maxRetries} failed:`, err.message);
        if (attempt === maxRetries) {
          addToActionLog(tabId, 'system', `Failed to reconnect after ${maxRetries} attempts`);
        }
      }
    }

    // All retries failed, wrap up and stop the task
    state.shouldStop = true;
    state.isRunning = false;
    state.completionStatus = 'wrapping_up';
    state.completedAt = Date.now();
    addToActionLog(tabId, 'system', 'Could not reconnect, wrapping up...');
    updateStoredState();

    // IMMEDIATELY remove blocker so user can interact with page
    await removeInteractionBlocker(tabId);

    // Request final summary (don't block user interaction)
    await requestFinalSummary(tabId, state);

    state.completionStatus = 'stopped';
    updateStoredState();
  } else if (!state.isRunning) {
    // Task not running, just clean up
    await removeInteractionBlocker(tabId);
  } else if (state.isRunning) {
    // Intentional detach (user canceled, tab closed, or DevTools opened) - wrap up the task
    state.shouldStop = true;
    state.isRunning = false;
    state.completionStatus = 'wrapping_up';
    state.completedAt = Date.now();
    addToActionLog(tabId, 'system', `Debugger detached (${reason}), wrapping up...`);
    updateStoredState();

    // IMMEDIATELY remove blocker so user can interact with page
    await removeInteractionBlocker(tabId);

    // Request final summary (don't block user interaction)
    await requestFinalSummary(tabId, state);

    state.completionStatus = 'stopped';
    updateStoredState();
  }
});

chrome.tabs.onRemoved.addListener((tabId) => {
  const state = agentStates.get(tabId);
  if (state) {
    if (state.debuggerAttached) chrome.debugger.detach({ tabId }).catch(() => {});
    agentStates.delete(tabId);
    lastClickCoords.delete(tabId);
    updateStoredState();
  }
  // Clean up caches even if no agent state exists for this tab
  lastClickCoords.delete(tabId);
  viewportCache.delete(tabId);
  pendingDelays.delete(tabId);
  // Clean up completed tasks history for this tab
  completedTasksHistory.delete(tabId);
});

// Invalidate viewport cache when window is resized
chrome.windows.onBoundsChanged.addListener(async (window) => {
  try {
    const tabs = await chrome.tabs.query({ windowId: window.id });
    for (const tab of tabs) {
      if (tab.id) viewportCache.delete(tab.id);
    }
  } catch (e) {
    // Window may have been closed
  }
});

// Invalidate viewport cache when zoom level changes
chrome.tabs.onZoomChange.addListener((zoomChangeInfo) => {
  viewportCache.delete(zoomChangeInfo.tabId);
});

// Clean up on startup - remove any leftover overlays from previous sessions
(async () => {
  chrome.storage.local.remove(['agentState', 'agentStates']);

  // Clean up overlays on all tabs in case extension was closed mid-task
  try {
    const tabs = await chrome.tabs.query({});
    for (const tab of tabs) {
      if (tab.id && tab.url && !tab.url.startsWith('chrome://') && !tab.url.startsWith('chrome-extension://')) {
        chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: pageCleanupFunction
        }).catch(() => {}); // Ignore errors for tabs we can't access
      }
    }
  } catch (e) {
    debugLog('[N1] Startup cleanup error:', e);
  }
})();
